/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Contains the all-important Vector3f class
 *
 * Revision History:
 * 
 *
 *--*/

using System;
using System.ComponentModel;

using BenWatson.BRayTracer;

namespace BenWatson.BRayTracer
{
	/// <summary>
	/// Vector3f describes a vector in 3 dimensions with floating point numbers.
	/// </summary>
	/// <remarks>There are two types of operations: static operators and instance operations.
	/// The static operators always create new vectors to store the result, but instance
	/// operations store the result in the this Vector3f.</remarks>
	[Serializable, TypeConverter(typeof(ExpandableObjectConverter))]
	public class Vector3f : ICloneable
	{
		/// <summary>
		/// X coordinate
		/// </summary>
		public float x;
		/// <summary>
		/// Y coordinate
		/// </summary>
		public float y;
		/// <summary>
		/// Z coordinate
		/// </summary>
		public float z;
		

		#region Properties
		
		/// <summary>
		/// Gets the magnitude of this vector
		/// </summary>
		[CategoryAttribute("Stats"), Description("he magnitude of this vector")]
		public float Magnitude 
		{
			get 
			{
				return (float)System.Math.Sqrt((x*x)+(y*y)+(z*z));
			}
		}

		/// <summary>
		/// Gets the square of the magnitude of this vector
		/// </summary>
		[CategoryAttribute("Stats"), Description("The magnitude-squared of this vector.")]
		public float MagnitudeSquared 
		{
			get 
			{
				return (x*x)+(y*y)+(z*z);
			}
		}
		
		/// <summary>
		/// Gets or sets the X coordinate
		/// </summary>
		[CategoryAttribute("General"), Description("X coordinate of vector")]
		public float X 
		{
			get 
			{
				return x;
			}
			set 
			{
				x=value;
			}
		}

		/// <summary>
		/// Gets or sets the Y coordinate
		/// </summary>
		[CategoryAttribute("General"), Description("Y coordinate of vector")]
		public float Y 
		{
			get 
			{
				return y;
			}
			set 
			{
				y=value;
			}
		}

		/// <summary>
		/// Gets or sets the Z coordinate
		/// </summary>
		[CategoryAttribute("General"), Description("Z coordinate of vector")]
		public float Z 
		{
			get 
			{
				return z;
			}
			set 
			{
				z=value;
			}
		}


		#endregion
		
		/// <summary>
		/// Initializes a new vector at the origin
		/// </summary>
		public Vector3f() 
		{
			x=y=z=0.0f;
			
		}
		/// <summary>
		/// Initializes a new vector from the given coordinates
		/// </summary>
		/// <param name="x">x coordinate</param>
		/// <param name="y">y coordinate</param>
		/// <param name="z">z coordinate</param>
		public Vector3f(float x, float y, float z) 
		{
			this.x = x;
			this.y = y;
			this.z = z;
		}

		/// <summary>
		/// Initializes a new vector from the given coordinates
		/// </summary>
		/// <param name="XYZ">Array of three values, {x,y,z}</param>
		/// <remarks>The array parameter must contain 3 values or an OutOfBoundsException will be thrown</remarks>
		public Vector3f(float[] XYZ) 
		{
			x = XYZ[0];
			y = XYZ[1];
			z = XYZ[2];
		}

		/// <summary>
		/// Normalizes this vector to unit length of 1
		/// </summary>
		public void Normalize() 
		{
            float lengthSquared = (x*x)+(y*y)+(z*z);
			if (lengthSquared==1.0 || lengthSquared == 0.0)
				return;
			else 
			{
				float length =(float)System.Math.Sqrt(lengthSquared);
				x/=length;
				y/=length;
				z/=length;
			}
		}

		#region static methods
		/// <summary>
		/// Returns a normalized version of the given vector. 
		/// </summary>
		/// <param name="Vector">The original vector</param>
		/// <returns>A vector with the same direction as Vector, with length 1.0</returns>
		public static Vector3f Normalize(Vector3f Vector) 
		{
			Vector3f v = (Vector3f)Vector.Clone();
			v.Normalize();
			return v;
		}

		/// <summary>
		/// Calculates cross product of two vectors
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector</param>
		/// <returns>A vector that is 90 degrees incident from both source vectors (cross product)</returns>
		public static Vector3f Cross(Vector3f Vector1, Vector3f Vector2) 
		{
			Vector3f v = new Vector3f(
										((Vector1.y*Vector2.z) - (Vector1.z*Vector2.y)),
										((Vector1.z*Vector2.x) - (Vector1.x*Vector2.z)),
										((Vector1.x*Vector2.y) - (Vector1.y*Vector2.x)) );
		
			return v;
		}

		/// <summary>
		/// Calculates the surface normal of a surface that includes the three specified points
		/// </summary>
		/// <param name="V1">First Vertex</param>
		/// <param name="V2">Second Vertex</param>
		/// <param name="V3">Third Vertex</param>
		/// <returns>A vector representing the surface normal</returns>
		public static Vector3f Normal(Vector3f V1, Vector3f V2, Vector3f V3) 
		{
			Vector3f v12 = V2-V1;
			Vector3f v23 = V3-V1;

			v12.Cross(v23);
			v12.Normalize();
			return v12;
		}

		/// <summary>
		/// Calculates dot product of two vectors
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector</param>
		/// <returns>Dot product of Vector1 and Vector2</returns>
		public static float Dot(Vector3f Vector1, Vector3f Vector2) 
		{
			return (Vector1.x*Vector2.x)+(Vector1.y*Vector2.y)+(Vector1.z*Vector2.z);
		}

		#endregion

		/// <summary>
		/// Calculates the Cross product of this vector and Vector2 and stores the results
		/// in this vector.
		/// </summary>
		/// <param name="Vector2">Second vector</param>
		public void Cross(Vector3f Vector2) 
		{
			float _x = ((y*Vector2.z) - (z*Vector2.y));
			float _y = ((z*Vector2.x) - (x*Vector2.z));
			float _z = ((x*Vector2.y) - (y*Vector2.x));

			x=_x;
			y=_y;
			z=_z;

		}

		#region Static Operators
		/// <summary>
		/// Adds two vectors together and returns the result
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector</param>
		/// <returns>Vector1+Vector2</returns>
		public static Vector3f operator+(Vector3f Vector1, Vector3f Vector2) 
		{
			return new Vector3f(Vector1.x+Vector2.x,Vector1.y+Vector2.y,Vector1.z+Vector2.z);
		}

		/// <summary>
		/// Negates the vector
		/// </summary>
		/// <param name="Vector">Vector to negate</param>
		/// <returns>-Vector (negative Vector)</returns>
		public static Vector3f operator-(Vector3f Vector) 
		{
			return new Vector3f(-Vector.x,-Vector.y,-Vector.z);
		}

		/// <summary>
		/// Subtracts Vector2 from Vector1
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector</param>
		/// <returns>Vector1 - Vector2</returns>
		public static Vector3f operator-(Vector3f Vector1, Vector3f Vector2) 
		{
            return new Vector3f(Vector1.x-Vector2.x,Vector1.y-Vector2.y,Vector1.z-Vector2.z);
		}

		/// <summary>
		/// Multiplies a vector by a scalar
		/// </summary>
		/// <param name="Vector">Vector argument</param>
		/// <param name="Scalar">Scalar to multiply with vector</param>
		/// <returns>A vector that equals Vector * Scalar</returns>
		public static Vector3f operator*(Vector3f Vector, float Scalar) 
		{
			return new Vector3f(Vector.x * Scalar, Vector.y * Scalar, Vector.z * Scalar);
		}

		/// <summary>
		/// Divides a vector by a scalar
		/// </summary>
		/// <param name="Vector">Vector argument</param>
		/// <param name="Scalar">Scalar to divide vector</param>
		/// <returns>A vector that equals Vector / Scalar</returns>
		public static Vector3f operator/(Vector3f Vector, float Scalar) 
		{
			return new Vector3f(Vector.x/Scalar,Vector.y/Scalar,Vector.z/Scalar);
		}
/*
		/// <summary>
		/// Determines if two vectors are equal
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector</param>
		/// <returns>True if vectors are equal, false otherwise</returns>
		public static bool operator==(Vector3f Vector1, Vector3f Vector2) 
		{
			return ((Vector1.X == Vector2.X) && (Vector1.Y==Vector2.Y) && (Vector1.Z==Vector2.Y));
		}

		/// <summary>
		/// Determines if two vectors are equal
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector (array of at least 3 floats; others are ignored)</param>
		/// <returns>True if vectors are equal, false otherwise. Returns false if Vector2 has less than 3 elements</returns>
		public static bool operator==(Vector3f Vector1, float[] Vector2) 
		{
			if (Vector2.Length < 3)
				return false;
			return ((Vector1.X==Vector2[0]) && (Vector1.Y == Vector2[1]) && (Vector1.Z == Vector2[2]));
		}

		/// <summary>
		/// Determines if two vectors are not equal
		/// </summary>
		/// <param name="Vector1"></param>
		/// <param name="Vector2"></param>
		/// <returns></returns>
		public static bool operator!=(Vector3f Vector1, Vector3f Vector2) 
		{
			return ((Vector1.X != Vector2.X) || (Vector1.Y!=Vector2.Y) || (Vector1.Z!=Vector2.Y));
		}

		/// <summary>
		/// Determines if two vectors are not equal
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector (array of at least 3 floats; others are ignored)</param>
		/// <returns>True if vectors are not equal, false otherwise. Returns true if Vector2 has less than 3 elements</returns>
		public static bool operator!=(Vector3f Vector1, float[] Vector2) 
		{
			if (Vector2.Length < 3)
				return true;
			return !((Vector1.X==Vector2[0]) && (Vector1.Y == Vector2[1]) && (Vector1.Z == Vector2[2]));
		}
*/
		#endregion

		/// <summary>
		/// Computes the angle between two vectors in radians
		/// </summary>
		/// <param name="Vector1">First vector</param>
		/// <param name="Vector2">Second vector</param>
		/// <returns>Angle in radians</returns>
		public static float AngleRadians(Vector3f Vector1, Vector3f Vector2) 
		{
			float dot = Dot(Vector1,Vector2);
			//float vectorsMag = (float)Math.Sqrt(Vector1.MagnitudeSquared * Vector2.MagnitudeSquared);
			float vectorsMag = Vector1.Magnitude * Vector2.Magnitude;

			double angle = System.Math.Acos(dot / vectorsMag);
			if (Double.IsNaN(angle))
				return 0.0f;
			
			return (float)angle;
		}


		#region Object Operators
		/// <summary>
		/// Adds the given vector to this vector and updates this vector
		/// </summary>
		/// <param name="Vector">The vector add to this one</param>
		public void Add(Vector3f Vector) 
		{
			x+=Vector.x;
			y+=Vector.y;
			z+=Vector.z;
		}

		/// <summary>
		/// Adds a constant to each component of this vector
		/// </summary>
		/// <param name="Constant">Constant to add to this vector</param>
		public void Add(float Constant) 
		{
			x+=Constant;
			y+=Constant;
			z+=Constant;
		}

		/// <summary>
		/// Substracts a Vector from this vector
		/// </summary>
		/// <param name="Vector">The vector to subtract</param>
		public void Subtract(Vector3f Vector) 
		{
			x-=Vector.x;
			y-=Vector.y;
			z-=Vector.z;
		}

		/// <summary>
		/// Subtracts a constant from each coordinate in this vector
		/// </summary>
		/// <param name="Constant">The constant</param>
		public void Subtract(float Constant) 
		{
			x-=Constant;
			y-=Constant;
			z-=Constant;
		}

		/// <summary>
		/// Multiplies the vector by a scalar
		/// </summary>
		/// <param name="Scalar">The scalar</param>
		public void Multiply(float Scalar) 
		{
			x*=Scalar;
			y*=Scalar;
			z*=Scalar;
		}

		/// <summary>
		/// Divides the vector by a scalar
		/// </summary>
		/// <param name="Scalar">The scalar</param>
		public void Divide(float Scalar) 
		{
			x/=Scalar;
			y/=Scalar;
			z/=Scalar;
		}

		#endregion

		#region Object overrides


		/// <summary>
		/// Converts this vector to its string representation
		/// </summary>
		/// <returns>String representation of the vector</returns>
		public override string ToString()
		{
			return String.Format("[{0:F2} {1:F2} {2:F2}]",x,y,z);
		}
		#endregion

		#region ICloneable Members

		/// <summary>
		/// Returns a copy of this vector
		/// </summary>
		/// <returns>A new vector</returns>
		public object Clone()
		{
			return new Vector3f(x,y,z);
		}

		#endregion

/*
		#region ICustomTypeDescriptor Members

		public TypeConverter GetConverter()
		{
			
			return TypeDescriptor.GetConverter(this,true);
		}

		public EventDescriptorCollection GetEvents(Attribute[] attributes)
		{
			return TypeDescriptor.GetEvents(this, attributes,true);
		}

		EventDescriptorCollection System.ComponentModel.ICustomTypeDescriptor.GetEvents()
		{
			return TypeDescriptor.GetEvents(this,true);
		}

		public string GetComponentName()
		{
			return TypeDescriptor.GetComponentName(this,true);
		}

		public object GetPropertyOwner(PropertyDescriptor pd)
		{
			return this;
		}

		public AttributeCollection GetAttributes()
		{
			return TypeDescriptor.GetAttributes(this,true);
		}

		public PropertyDescriptorCollection GetProperties(Attribute[] attributes)
		{
			return TypeDescriptor.GetProperties();
		}

		PropertyDescriptorCollection System.ComponentModel.ICustomTypeDescriptor.GetProperties()
		{
			PropertyDescriptorCollection pds = new PropertyDescriptorCollection(null);
			
		}

		public object GetEditor(Type editorBaseType)
		{
			return TypeDescriptor.GetEditor(this,editorBaseType,true);
		}

		public PropertyDescriptor GetDefaultProperty()
		{
			return TypeDescriptor.GetDefaultProperty(this,true);
		}

		public EventDescriptor GetDefaultEvent()
		{
			return TypeDescriptor.GetDefaultEvent(this,true);
		}

		public string GetClassName()
		{
			return TypeDescriptor.GetClassName(this,true);
		}

		#endregion
		*/
	}
	
}
