/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *		Contains statistics for the ray tracer to track
 *     
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added comments
 *		12/20/2004 - Ben Watson - Added to VSS
 * 
 *
 *--*/

using System;

namespace BenWatson.BRayTracer
{
	/// <summary>
	/// Summary description for Stats.
	/// </summary>
	public struct Stats
	{
		/// <summary>
		/// The number of rays traced so far
		/// </summary>
		public static int NumberOfRaysTraced;

		/// <summary>
		/// The number of lines (of the output image) traced so far 
		/// </summary>
		public static int NumberOfLinesTraced;
		/// <summary>
		/// The number of pixels in the output image
		/// </summary>
		public static int NumberOfPixels;
		/// <summary>
		/// The total number of lines in the output image
		/// </summary>
		public static int TotalLines;

		#region Private Fields
		/// <summary>
		/// Start time
		/// </summary>
		private static DateTime m_startTime;
		/// <summary>
		/// End time
		/// </summary>
		private static DateTime m_endTime;
		/// <summary>
		/// Elapsed time so far
		/// </summary>
		private static TimeSpan m_elapsedTime;
		#endregion

		/// <summary>
		/// Gets or sets the start time
		/// </summary>
		public static DateTime StartTime 
		{
			get 
			{
				return m_startTime;
			}
			set 
			{
				m_startTime = value;
			}
		}

		/// <summary>
		/// Gets or sets the finish time
		/// </summary>
		public static DateTime FinishTime 
		{
			get 
			{
				return m_endTime;
			}
			set 
			{
				m_endTime = value;
				m_elapsedTime = m_endTime-m_startTime;
			}
		}

		/// <summary>
		/// Gets the elapsed time from start to finish
		/// </summary>
		public static TimeSpan ElapsedTime 
		{
			get 
			{
				return m_elapsedTime;
			}
		}
		/// <summary>
		/// Gets the number of rays traced per second
		/// </summary>
		public static float RaysPerSecond 
		{
			get 
			{
				return (float)NumberOfRaysTraced / (float)ElapsedTime.Seconds;
			}
		}

		/// <summary>
		/// Resets stats to pre-render state
		/// </summary>
		public static void Reset() 
		{
			NumberOfRaysTraced=0;
			NumberOfLinesTraced=0;
			TotalLines=0;
		}
	}
}
