/*++
 * Copyright 2004 Ben Watson. 
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     This file contains the implementation for the Tracer class. Tracer implements
 * the main raytracing functionality by simulating the dispersal of light rays from the
 * viewpoint to each object in the scene.
 *
 * Revision History:
 * 
 *		12/20/2004 - Added existing file to source control
 *		12/26/2004 - Added XML comments
 *		12/26/2004 - Added max depth testing
 *
 *--*/


using System;
using System.Collections;
using System.Drawing;
using BenWatson.BRayTracer.Primitives;


namespace BenWatson.BRayTracer.Raytracer
{
	/// <summary>
	/// Tracer implements the main ray tracing functionality. It contains functions for calculating rays,
	/// getting the color at a point, etc.
	/// </summary>
	public class Tracer
	{
		#region Private Fields
		/// <summary>
		/// The scene which we're rendering
		/// </summary>
		private Scene m_scene = null;
		/// <summary>
		/// The settings that define how we render
		/// </summary>
		private RayTraceSettings m_settings =new RayTraceSettings();
		/// <summary>
		/// Width of the view in sub-pixels
		/// </summary>
		private int m_widthAA=0;
		/// <summary>
		/// Height of the view in sub-pixels
		/// </summary>
		private int m_heightAA=0;
		/// <summary>
		/// Maximum number of recursions to follow in tracing a ray
		/// </summary>
		private int m_traceLevels=999;

		/// <summary>
		/// Temporary list of shapes, after some have been culled
		/// </summary>
		private ArrayList m_shapes = null;

		#endregion

		/// <summary>
		/// Gets or sets the settings for ray tracing
		/// </summary>
		private RayTraceSettings Settings 
		{
			get 
			{
				return m_settings;
			}
			set 
			{
				m_settings = value;
				m_widthAA = m_settings.ImageWidth * (int)Math.Sqrt(m_settings.SubPixels);
				m_heightAA = m_settings.ImageHeight * (int)Math.Sqrt(m_settings.SubPixels);
				
				m_traceLevels = m_settings.TraceLevels;

			}
		}

		/// <summary>
		/// Initializes a Tracer object from the given settings and a scene
		/// </summary>
		/// <param name="Settings">The tracer settings</param>
		/// <param name="Scene">The scene to render</param>
		public Tracer(RayTraceSettings Settings, Scene Scene)
		{
			this.Settings = Settings;
			
			this.m_scene = Scene;

			//listen for changes in the settings
			m_settings.ImageSizeChanged+=new EventHandler(m_settings_ImageSizeChanged);
			m_settings.TraceLevelsChanged+=new EventHandler(m_settings_TraceLevelsChanged);
			m_settings.SubPixelsChanged+=new EventHandler(m_settings_SubPixelsChanged);
			
		}

		/// <summary>
		/// Performs calculations on each scene object to save future computation
		/// </summary>
		public void DoPreRenderCalculations() 
		{
			//get list of shapes, remove objects that are too far
			this.m_shapes = new ArrayList();

			foreach (Shape s in m_scene) 
			{
				Vector3f dV = m_scene.Camera.Position - s.Position;
				float dist = dV.Magnitude;

				if (dist <= this.m_settings.MaxDepth) 
				{
					m_shapes.Add(s);
					s.DoPreRenderCalculations();
				}
			}
		}

		/// <summary>
		/// Returns the color of a primary ray (level 0 recursion)
		/// </summary>
		/// <param name="ray">The primary ray to trace</param>
		/// <returns>The color of the ray</returns>
		public Color4f GetColorOfRay(Ray ray) 
		{
            return GetColorOfRay(ray,0);
		}

		/// <summary>
		/// Returns the color of a ray
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <param name="level">Recursion level</param>
		/// <returns>The color of the ray</returns>
		private Color4f GetColorOfRay(Ray ray, int level) 
		{
			//update stats
			Stats.NumberOfRaysTraced++;

			//find first object it hits
			Shape closestShape = null;
			float closestShapeDist = float.MaxValue;

			
			for (int i=0;i<m_shapes.Count;i++) 
			{
				Shape s=(Shape)m_shapes[i];
				float dist = s.DistanceToHit(ray);
				if (dist>0.0f && dist < closestShapeDist) 
				{
					closestShapeDist = dist;
					closestShape=s;
				}
			}

			
			if (closestShape!=null) 
			{
				//calculate color for this point on the object
				Vector3f intersectionPoint = (Vector3f)ray.Start.Clone();
				intersectionPoint.Add(ray.Direction *(0.999f * closestShapeDist));

				return GetColorOfRayFromLight(closestShape,intersectionPoint,ray.Direction, level);
			}

			//no objects hit, just return background color
			return m_scene.BackgroundColor;

		}

		/// <summary>
		/// Gets the diffuse color of the first object the ray hits
		/// </summary>
		/// <param name="ray">Ray to trace</param>
		/// <returns>A color</returns>
		public Color4f GetQuickColorOfRay(Ray ray) 
		{
			foreach(Shape s in m_scene) 
			{
				if (s.IsHit(ray))
					return s.Material.DiffuseColor;
			}
			return m_scene.BackgroundColor;
		}

		/// <summary>
		/// Gets the color of a ray at a specific point on a specific shape
		/// </summary>
		/// <param name="S">The shape that was hit</param>
		/// <param name="Point">The point at the shape (in WC)</param>
		/// <param name="RayDirection">The direction of the ray (in WC)</param>
		/// <param name="level">The recursion level</param>
		/// <returns>The color of the ray</returns>
		private Color4f GetColorOfRayFromLight(Shape S, Vector3f Point, Vector3f RayDirection, int level) 
		{
			//TODO: optimize math in this function
			Material mat = S.GetMaterialAtPoint(Point);
			Color4f outputColor = new Color4f(0.0f,0.0f,0.0f);

			Vector3f surfaceNormal = S.NormalAtPoint(Point);

			float negRayDirOnNormal = -Vector3f.Dot(RayDirection,surfaceNormal);
			Vector3f reflectRayDir = RayDirection;
			surfaceNormal.Multiply(2*negRayDirOnNormal);
			//Vector3f temp = surfaceNormal * (2*negRayDirOnNormal);
			
			reflectRayDir.Add(surfaceNormal);
			surfaceNormal.Divide(2*negRayDirOnNormal);

			//get reflected color
			if (mat.Reflectivity!=0.0f && level<m_traceLevels) 
			{
				Ray reflectedRay = new Ray(Point,reflectRayDir);
				Color4f reflectedColor = (Color4f)GetColorOfRay(reflectedRay,level+1).Clone();

				reflectedColor.Multiply(mat.Reflectivity);
				outputColor.Add(reflectedColor);
			}


			//get non-reflected color

			Color4f nonReflectedLight = new Color4f(mat.DiffuseColor.Red,mat.DiffuseColor.Green,mat.DiffuseColor.Blue);

			//add in ambient light component (ambient * diffuse color)
			//nonReflectedLight+=(mat.DiffuseColor * m_scene.AmbientLight);
			nonReflectedLight.Multiply(m_scene.AmbientLight);
			
			//get color due to lights
			Color4f colorFromLight = new Color4f();
			foreach(Light L in m_scene.Lights) 
			{
				colorFromLight.Red=0.0f;
				colorFromLight.Green=0.0f;
				colorFromLight.Blue=0.0f;
                
				Vector3f lightToPoint = Point - L.Position;
				float dot = -Vector3f.Dot(lightToPoint, surfaceNormal);
				if (dot<0.0f)//surface facing away
					continue;
				float dist = lightToPoint.Magnitude;
				/* *
				 * Inverse square law is technically correct, but inverse looks better
				 * */
				//float inverseLightDistSquared = 1.0f/(dist*dist);
				float inverseLightDistSquared = 1.0f/(dist);
				lightToPoint.Normalize();
				dot/=dist;

				//see if there is something blocking this ray
				if (RaySegmentHitsAnything(new Ray(Point, lightToPoint * -1.0f), dist)) 
					continue;

                //modify diffuse color due to lights
				float diffuseIntensity = dot * inverseLightDistSquared;
				
                colorFromLight.Red = mat.DiffuseColor.Red * diffuseIntensity;
				colorFromLight.Green = mat.DiffuseColor.Green * diffuseIntensity;
				colorFromLight.Blue = mat.DiffuseColor.Blue * diffuseIntensity;
				
				//specular
				if (mat.SpecularAmount != 0.0f) 
				{
					float specDot = -Vector3f.Dot(reflectRayDir,lightToPoint);
					
					if (specDot > 0.0f) 
					{
						float specularIntensity = specDot * inverseLightDistSquared * 
							(float)System.Math.Pow(specDot,mat.SpecularCoefficient) * mat.SpecularAmount;

						colorFromLight.Add(specularIntensity);
					}
				}
				colorFromLight.Multiply(L.Color);
				nonReflectedLight .Add(colorFromLight);
			} //end foreach light

			//combine non reflected light to output
			nonReflectedLight.Multiply(1.0f-mat.Reflectivity);
			outputColor.Add(nonReflectedLight);

			/* Transmitted light
			 * 
			 * */

			if (mat.Opacity < 1.0) 
			{
				Color4f transmittedLight = new Color4f(0.0f,0.0f,0.0f);

				//calculate transmitted ray
				//TODO: assign materials this number
				float n1 = 1.0f;//refraction index of incident ray
				float n2 = 1.0f;//refraction index of medium

				float nDotI = Vector3f.Dot(surfaceNormal,RayDirection);

				float cosTheta = (float)Math.Sqrt(1-(n1/n2)*(n1/n2)*
					(1-nDotI * nDotI));

				Vector3f T = RayDirection*(n1/n2) -  surfaceNormal*(cosTheta + (n1/n2)*nDotI) ;

                Ray tRay = new Ray((Point / 0.999f) * 1.0001f, T);
				Color4f tColor = GetColorOfRay(tRay, level+1);

				transmittedLight = tColor * (1.0f-mat.Opacity);
				outputColor = outputColor * mat.Opacity + transmittedLight;

			}

			return outputColor;
		
		}

		/// <summary>
		/// Determines if a ray hits any shape closer than the given distance
		/// </summary>
		/// <param name="ray">The ray to test</param>
		/// <param name="Distance">The maximum distance to test for</param>
		/// <returns>True if the ray hits a shape closer than Distance; false otherwise</returns>
		/// <remarks>This function is used in shadow calculations, to see if a point is hit by a light
		/// or if it's masked by another shape</remarks>
		private bool RaySegmentHitsAnything(Ray ray, float Distance) 
		{
			ArrayList shapes = m_scene.Shapes;
			for (int i=0;i<shapes.Count;i++) 
			{
				Shape s=(Shape)shapes[i];
				float dist = s.DistanceToHit(ray);
				if (dist>=0.0f && dist < Distance) 
				{
					return true;
				}
			}
			return false;
		}

		/// <summary>
		/// Calculates the ray from the camera's position to the specified pixel
		/// </summary>
		/// <param name="X">X coordinate of pixel</param>
		/// <param name="Y">Y coordinate of pixel</param>
		/// <returns>Ray from camera position to screen pixel</returns>
		public Ray GetRayFromPixel(int X, int Y) 
		{
			
			float xFrac = (float)X/m_settings.ImageWidth;
			float yFrac = (float)Y/m_settings.ImageHeight;
			Vector3f dir = new Vector3f((xFrac-0.5f)*m_settings.AspectRatio,-(yFrac-0.5f),1.0f);

			//float x0 = X / m_settings.ImageWidth - 0.5f;
			//float y0 = ((m_settings.ImageHeight - 1) - Y) / m_settings.ImageHeight - 0.5f;

			
			return new Ray(m_scene.Camera.Position, dir);
		}

		/// <summary>
		/// Calculates the ray from the camera's position to the specified subpixel
		/// </summary>
		/// <param name="X">X coordinate of pixel</param>
		/// <param name="Y">Y coordinate of pixel</param>
		/// <param name="subX">X coordinate of subpixel</param>
		/// <param name="subY">Y coordinate of subpixel</param>
		/// <returns>Ray from camera position to screen pixel</returns>
		public Ray GetRayFromPixelAA(int X, int Y, int subX, int subY) 
		{
			int subSize = (int)Math.Sqrt(m_settings.SubPixels);
			float xFrac = (float)(X*subSize+subX) /(m_widthAA);
			float yFrac = (float)(Y*subSize+subY) /(m_heightAA);

			Vector3f dir = new Vector3f((xFrac-0.5f) * m_settings.AspectRatio, -(yFrac-0.5f),1.0f);

			return new Ray(m_scene.Camera.Position, dir);
		}

		/// <summary>
		/// Event handler for changed image size
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void m_settings_ImageSizeChanged(object sender, EventArgs e)
		{
			m_widthAA = m_settings.ImageWidth * (int)Math.Sqrt(m_settings.SubPixels);
			m_heightAA = m_settings.ImageHeight * (int)Math.Sqrt(m_settings.SubPixels);
		}

		/// <summary>
		/// Event handler for changed maximum recursion level
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void m_settings_TraceLevelsChanged(object sender, EventArgs e)
		{
			m_traceLevels = m_settings.TraceLevels;
		}

		/// <summary>
		/// Event handler for changed subpixel amount (for antialiasing)
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void m_settings_SubPixelsChanged(object sender, EventArgs e)
		{
			m_widthAA = m_settings.ImageWidth * (int)Math.Sqrt(m_settings.SubPixels);
			m_heightAA = m_settings.ImageHeight * (int)Math.Sqrt(m_settings.SubPixels);
		}
	}
}
