/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *		Describes an entire ray tracing scene.     
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/26/2004 - Ben Watson - Added XML comments
 * 
 *
 *--*/

using System;
using System.Drawing;
using System.Collections;

using BenWatson.BRayTracer.Primitives;

namespace BenWatson.BRayTracer.Raytracer
{
	/// <summary>
	/// Describes a virtual scene
	/// </summary>
	[Serializable]
	public class Scene : IEnumerable
	{
		#region Fields
		/// <summary>
		/// The camera
		/// </summary>
		private Camera m_camera;
		/// <summary>
		/// Collection of shapes
		/// </summary>
		private ArrayList m_shapes = new ArrayList(); 
		/// <summary>
		/// Collection of lights
		/// </summary>
		private ArrayList m_lights = new ArrayList();
		/// <summary>
		/// Filename from which the scene was loaded.
		/// </summary>
		/// <remarks>If this is null, then the file has not yet been saved.</remarks>
		[NonSerialized()]
		private String m_filename;
		/// <summary>
		/// Background color of scene.
		/// </summary>
		/// <remarks>The background color is the color of a ray when it doesn't hit anything.</remarks>
		private Color4f m_backgroundColor = new Color4f(0.0f,0.2f,0.2f);
		/// <summary>
		/// Ambient light of scene
		/// </summary>
		private Color4f m_ambientLight = new Color4f(0.5f,0.5f,0.5f);
		#endregion

		#region Properties
		/// <summary>
		/// Gets or sets the camera associated with this scene
		/// </summary>
		public Camera Camera 
		{
			get 
			{
				return m_camera;
			}
			set 
			{
				m_camera=value;
			}
		}

		/// <summary>
		/// Gets the number of shapes in this scene
		/// </summary>
		public int NumShapes 
		{
			get 
			{
				return m_shapes.Count;
			}
		}

		/// <summary>
		/// Gets or sets the filename associated with this scene
		/// </summary>
		public String Filename 
		{
			get 
			{
				return m_filename;
			}
			set 
			{
				m_filename = value;
			}
		}
/*
		/// <summary>
		/// True if scene is not saved; false otherwise
		/// </summary>
		public bool IsDirty 
		{
			get 
			{
				return m_isDirty;
			} 
			set 
			{
				m_isDirty=value;
			}
		}
		*/

		/// <summary>
		/// Gets or sets the scene's background color
		/// </summary>
		public Color4f BackgroundColor 
		{
			get 
			{
				return this.m_backgroundColor;
			}
			set 
			{
				m_backgroundColor=value;

			}
		}

		/// <summary>
		/// Gets or sets the scene's ambient light
		/// </summary>
		public Color4f AmbientLight 
		{
			get 
			{
				return m_ambientLight;
			}
			set 
			{
				m_ambientLight = value;
			
			}
		}
		
		/// <summary>
		/// Gets array of lights
		/// </summary>
		public ArrayList Lights 
		{
			get 
			{
				return m_lights;
			}
		}
	
		/// <summary>
		/// Gets array of shapes
		/// </summary>
		public ArrayList Shapes 
		{
			get 
			{
				return m_shapes;
			}
		}
		#endregion

		/// <summary>
		/// Initializes a new, empty scene
		/// </summary>
		public Scene()
		{
			//
			// TODO: Add constructor logic here
			//
		}

		#region IEnumerable Members

		/// <summary>
		/// Returns an enumerator to the shapes
		/// </summary>
		/// <returns></returns>
		public IEnumerator GetEnumerator()
		{
			return m_shapes.GetEnumerator();
		}

		#endregion

		/// <summary>
		/// Adds a shape to the scene
		/// </summary>
		/// <param name="NewShape">Shape to add</param>
		public void AddShape(Shape NewShape) 
		{
			this.m_shapes.Add(NewShape);
			
		}

		/// <summary>
		/// Removes a shape from the scene
		/// </summary>
		/// <param name="ShapeToRemove">Shape to remove</param>
		public void RemoveShape(Shape ShapeToRemove) 
		{
			this.m_shapes.Remove(ShapeToRemove);
			
		}

		/// <summary>
		/// Adds a light to the scene
		/// </summary>
		/// <param name="NewLight">The light to add</param>
		public void AddLight(Light NewLight) 
		{
			this.m_lights.Add(NewLight);
			
		}

		/// <summary>
		/// Removes a light from the scene
		/// </summary>
		/// <param name="LightToRemove">Light to remove</param>
		public void RemoveLight(Light LightToRemove) 
		{
			this.m_lights.Remove(LightToRemove);
			
		}
	}
}
