/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implements a material for scene objects
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/26/2004 - Ben Watson - Added XML comments
 *		12/26/2004 - Ben Watson - changed private fields to protected. Added a Name field/property.
 *		12/26/2004 - Ben Watson - added opacity
 *
 *--*/

using System;
using System.ComponentModel;
using System.Drawing;

namespace BenWatson.BRayTracer.Raytracer
{
	/// <summary>
	/// Describes a shape's material properties
	/// </summary>
	[Serializable, TypeConverter(typeof(ExpandableObjectConverter))]
	public class Material : ICloneable
	{
		#region Fields
		/// <summary>
		/// Diffuse color
		/// </summary>
		protected Color4f m_diffuseColor = new Color4f(1.0f,0.0f,0.0f,1.0f);
		/// <summary>
		/// Specular color
		/// </summary>
		protected Color4f m_specularColor = new Color4f(1.0f,1.0f,1.0f,1.0f);
		/// <summary>
		/// Specular amount
		/// </summary>
		protected float m_specularAmount = 10.0f;
		/// <summary>
		/// Specular coefficient
		/// </summary>
		protected float m_specularCoefficient = 10.0f ;
		/// <summary>
		/// Amount of reflectivity
		/// </summary>
		protected float m_reflectivity = 0.5f;
		/// <summary>
		/// Name of the material
		/// </summary>
		protected string m_name = "";
		/// <summary>
		/// Percent of opacity
		/// </summary>
		protected float m_opacity = 1.0f;
		#endregion

		#region Properties
		
		/// <summary>
		/// Gets or sets this material's name
		/// </summary>
		[Viewable, CategoryAttribute("General"), Description("Name of the material")]
		public string Name 
		{
			get 
			{
				return m_name;
			}
			set 
			{
				m_name = value;
			}
		}
		/// <summary>
		/// Gets or sets the diffuse color
		/// </summary>
		[Editable, CategoryAttribute("Colors"), Description("Main (diffuse) color of object")]
		public Color4f DiffuseColor 
		{
			get 
			{
				return m_diffuseColor;
			}
			set 
			{
				m_diffuseColor = value;
			}
		}

		/// <summary>
		/// Gets or sets the specular highlight color
		/// </summary>
		[Editable, CategoryAttribute("Colors"), Description("Color of specular highlights")]
		public Color4f SpecularColor 
		{
			get 
			{
                return m_specularColor;
			}
			set 
			{
				m_specularColor = value;
			}
		}

		/// <summary>
		/// Gets or sets the amount of specularity (0.0 - infinity)
		/// Determines matte vs. glossy
		/// </summary>
		[Editable, Category("Specular"), Description("Amount of specularity (0.0 - infinitity). Higher values increase size of specular area.")]
		public float SpecularAmount 
		{
			get 
			{
				return m_specularAmount;
			}
			set 
			{
				m_specularAmount = value;
				if (m_specularAmount<0.0f)
					m_specularAmount=0.0f;
			}
		}

		/// <summary>
		/// Gets or sets the specular coefficient (0.0 - infinity)
		/// Determines shininess.
		/// </summary>
		[Editable, Category("Specular"), Description("Specular Coefficient (0.0 - infinity). A higher value means the specular highlight is more focused.")]
		public float SpecularCoefficient 
		{
			get 
			{
				return m_specularCoefficient;
			}
			set 
			{
				m_specularCoefficient = value;
				if (m_specularCoefficient<0.0f)
					m_specularCoefficient=0.0f;
			}
		}

		/// <summary>
		/// Gets or sets the reflectivity (0.0 - 1.0)
		/// </summary>
		/// <remarks>A reflectivity of 0.0 reflects no light. A reflectivity of 1.0 is a perfect mirror.</remarks>
		[Editable, Category("General"), Description("Amount of reflectivity of material (0.0 - 1.0). A reflectivity of 1.0 is a perfect mirror.")]
		public float Reflectivity 
		{
			get 
			{
				return m_reflectivity;
			}
			set 
			{
				m_reflectivity = value;
				if (m_reflectivity>1.0f)
					m_reflectivity = 1.0f;
				else if (m_reflectivity<0.0f)
					m_reflectivity=0.0f;
			}
		}

		/// <summary>
		/// Gets or sets the opacity (0.0 - 1.0)
		/// </summary>
		/// <remarks>An opacity of 0.0 means the material is completely clear (invisible). An opacity of 1.0 is not at all transparent.</remarks>
		[Editable, Category("General"), Description("Opacity of object (0.0 - 1.0). Opacity of 0.0 is completely transparent.")]
		public float Opacity 
		{
			get 
			{
				return m_opacity;
			}
			set
			{
				m_opacity = value;
				if (m_opacity>1.0f)
					m_opacity=1.0f;
				else if (m_opacity<0.0f)
					m_opacity=0.0f;
			}
		}

		#endregion

		/// <summary>
		/// Initializes a new material with red color, and white higlights
		/// </summary>
		public Material()
		{

		}

		/// <summary>
		/// Initializes a new material with the given diffuse color
		/// </summary>
		/// <param name="DiffuseColor">Diffuse color</param>
		public Material(Color4f DiffuseColor) 
		{
			m_diffuseColor = DiffuseColor;
		}

		/// <summary>
		/// Initializes a new material from the given diffuse and specular colors
		/// </summary>
		/// <param name="DiffuseColor">Diffuse color</param>
		/// <param name="SpecularColor">Specular color</param>
		public Material(Color4f DiffuseColor, Color4f SpecularColor) 
		{
			m_diffuseColor = DiffuseColor;
			m_specularColor = SpecularColor;
		}

		/// <summary>
		/// Initializes a new material from the given diffuse and specular colors and reflectivity
		/// </summary>
		/// <param name="DiffuseColor">Diffuse color</param>
		/// <param name="SpecularColor">Specular color</param>
		/// <param name="Reflectivity">Reflectivity (0.0 - 1.0)</param>
		public Material(Color4f DiffuseColor, Color4f SpecularColor, float Reflectivity) 
		{
			m_diffuseColor = DiffuseColor;
			m_specularColor = SpecularColor;
			this.Reflectivity = Reflectivity;
		}

		/// <summary>
		/// Initializes a new material from the given diffuse and specular colors
		/// </summary>
		/// <param name="DiffuseColor">Diffuse color</param>
		/// <param name="SpecularColor">Specular color</param>
		/// <param name="Reflectivity">Reflectivity (0.0 - 1.0)</param>
		/// <param name="SpecularAmount">Specular amount (0.0 - infinity)</param>
		/// <param name="SpecularCoefficient">Specular coefficient (0.0 - infinity)</param>
		public Material(Color4f DiffuseColor, Color4f SpecularColor, float Reflectivity, 
			float SpecularAmount, float SpecularCoefficient) 
		{
			m_diffuseColor = DiffuseColor;
			m_specularColor = SpecularColor;
			this.Reflectivity = Reflectivity;
			this.SpecularAmount = SpecularAmount;
			this.SpecularCoefficient = SpecularCoefficient;
		}

		/// <summary>
		/// Gets a string representation of this material
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			return m_name;
		}

		#region ICloneable Members

		/// <summary>
		/// Returns a copy of this material
		/// </summary>
		/// <returns>A copy of this material</returns>
		public object Clone()
		{
			Material mat = new Material();
			mat.m_diffuseColor=(Color4f)m_diffuseColor.Clone();
			mat.m_reflectivity = m_reflectivity;
			mat.m_specularAmount = m_specularAmount;
			mat.m_specularCoefficient = m_specularCoefficient;
			mat.m_specularColor = (Color4f)m_specularColor.Clone();

			return mat;
		}

		#endregion


		
	}
}

