/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implmentation of a point light source.
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/25/2004 - Ben Watson - Added XML comments
 *
 *--*/

using System;
using System.Drawing;
using BenWatson.BRayTracer;


namespace BenWatson.BRayTracer.Raytracer
{
	/// <summary>
	/// Represents a point light source
	/// </summary>
	[Serializable]
	public class Light : ICloneable
	{
		/// <summary>
		/// Light color
		/// </summary>
		protected Color4f m_color=new Color4f(1.0f,1.0f,1.0f);
		/// <summary>
		/// Light position
		/// </summary>
		protected Vector3f m_position;
		/// <summary>
		/// Light name
		/// </summary>
		protected String m_name="";

		#region Properties
		/// <summary>
		/// Gets or sets the color of this light
		/// </summary>
		[Editable]
		public Color4f Color 
		{
			get 
			{
				return m_color;
			}
			set 
			{
				m_color=value;
			}
		}

		/// <summary>
		/// Gets or sets the name of this light
		/// </summary>
		[Editable]
		public String Name 
		{
			get 
			{
				return m_name;
			}
			set 
			{
				m_name = value;
			}
		}

		/// <summary>
		/// Gets or sets the position of this light
		/// </summary>
		[Editable]
		public Vector3f Position 
		{
			get 
			{
				return m_position;
			}
			set 
			{
				m_position=value;
			}
		}
		#endregion
		
	
		/// <summary>
		/// Initializes a new point light source located at the origin
		/// </summary>
		public Light()
		{
			m_position = new Vector3f(0.0f,0.0f,0.0f);
			
		}

		/// <summary>
		/// Initializes a new point light source located at the position specified by the vector
		/// </summary>
		/// <param name="Position">Vector position of new light source</param>
		public Light(Vector3f Position) 
		{
			m_position = Position;
		}
		#region ICloneable Members

		/// <summary>
		/// Creates a copy of this light
		/// </summary>
		/// <returns>A copy of the light</returns>
		public object Clone()
		{
			Light newLight = new Light();
			newLight.m_color = this.m_color;
			newLight.m_position = (Vector3f)this.m_position.Clone();
			return newLight;
		}

		#endregion
	}
}
