/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Contains structures to control rendering:
 *		1) ImageSize
 *		2) RayTraceSettings
 *
 * Revision History:
 *		12/20/2004 Ben Watson - added XML comments
 *		12/20/2004 Ben Watson - added source to VSS
 *		12/26/2004 Ben Watson - changed ray trace settings to be a struct instead of class
 * 
 *
 *--*/

using System;

namespace BenWatson.BRayTracer
{

	/// <summary>
	/// A width, height pair for describing image sizes
	/// </summary>
	public struct ImageSize 
	{
		/// <summary>
		/// Width of the image
		/// </summary>
		public int Width ;

		/// <summary>
		/// Height of image
		/// </summary>
		public int Height;

		/// <summary>
		/// Constructs a new ImageSize object
		/// </summary>
		/// <param name="Width">Image width</param>
		/// <param name="Height">Image height</param>
		public ImageSize(int Width, int Height) 
		{
			this.Width=Width;
			this.Height=Height;
			
		}
		/// <summary>
		/// Returns a string representation of this image size
		/// </summary>
		/// <returns>A string representing this image size</returns>
		/// <remarks>Strings are in the format "Width x Height"</remarks>
		public override string ToString()
		{
			return Width.ToString()+" x " +Height.ToString();
		}

	}

	/// <summary>
	/// Settings for ray tracer
	/// </summary>
	[Serializable]
	public struct RayTraceSettings
	{
		#region Fields
		/// <summary>
		/// Antialiasing flag
		/// </summary>
		private bool m_antiAlias;
		/// <summary>
		/// Number of sub pixels
		/// </summary>
		private int m_subPixels;
		/// <summary>
		/// Image width
		/// </summary>
		private int m_imageWidth;
		/// <summary>
		/// Image height
		/// </summary>
		private int m_imageHeight;
		/// <summary>
		/// Aspect ratio
		/// </summary>
		private float m_aspectRatio;
		/// <summary>
		/// Maximum number of recursive trace levels
		/// </summary>
		private int m_traceLevels;

		/// <summary>
		/// Depth of maximum visible object
		/// </summary>
		private float m_maxDepth;
		#endregion

		#region Properties
		/// <summary>
		/// Gets or sets antialiasing status
		/// </summary>
		public bool AntiAlias 
		{
			get 
			{
				return m_antiAlias;
			}
			set 
			{
				m_antiAlias = value;
				if (AntiAliasChanged!=null)
					AntiAliasChanged(this,EventArgs.Empty);
			}
		}

		/// <summary>
		/// Gets or sets the number of subpixels across (and down)
		/// </summary>
		/// <remarks>Subpixels refers to the total number of sub-pixels. Thus, it is always a perfect square.
		/// For example, a value of 9 would refer to a block of 3 by 3 pixels.</remarks>
		public int SubPixels 
		{
			get 
			{
				return m_subPixels;
			}
			set 
			{
				m_subPixels = value;
				if (SubPixelsChanged!=null)
					SubPixelsChanged(this,EventArgs.Empty);
			}
		}

		/// <summary>
		/// Gets or sets the image width
		/// </summary>
		public int ImageWidth 
		{
			get 
			{
				return m_imageWidth;
			}
			set 
			{
				m_imageWidth=value;
				m_aspectRatio = (float)m_imageWidth/m_imageHeight;
				if (ImageSizeChanged!=null)
					ImageSizeChanged(this,EventArgs.Empty);
			}
		}

		/// <summary>
		/// Gets or sets the image height
		/// </summary>
		public int ImageHeight 
		{
			get 
			{
				return m_imageHeight;
			}
			set 
			{
				m_imageHeight=value;
				m_aspectRatio=(float)m_imageWidth/(float)m_imageHeight;
				if (ImageSizeChanged!=null)
					ImageSizeChanged(this,EventArgs.Empty);
			}
		}

		/// <summary>
		/// Gets the aspect ratio
		/// </summary>
		public float AspectRatio 
		{
			get {
				return m_aspectRatio;
			}
		}
	
		/// <summary>
		/// Gets or sets the maximum number of recursive tracing levels
		/// </summary>
		public int TraceLevels 
		{
			get 
			{
				return m_traceLevels;
			}
			set 
			{
				m_traceLevels=value;
				if (TraceLevelsChanged!=null)
				{
					TraceLevelsChanged(this,EventArgs.Empty);
				}
			}
		}

		/// <summary>
		/// Depth of maximum visible object
		/// </summary>
		public float MaxDepth 
		{
			get 
			{
				return m_maxDepth;
			}
			set 
			{
				m_maxDepth = value;
				if (MaxDepthChanged!=null) 
					MaxDepthChanged(this,EventArgs.Empty);
			}
		}
		#endregion

		#region Events
		/// <summary>
		/// Notifies when the image's size has changed
		/// </summary>
		public event EventHandler ImageSizeChanged;
		/// <summary>
		/// Notifies when the number of subpixels has changed
		/// </summary>
		public event EventHandler SubPixelsChanged;
		/// <summary>
		/// Notifies when the antialiasing has been turned on or off
		/// </summary>
		public event EventHandler AntiAliasChanged;
		/// <summary>
		/// Notifies when the maximum number of recursive trace levels has changed
		/// </summary>
		public event EventHandler TraceLevelsChanged;
		/// <summary>
		/// Notifies when the maximum depth of traced objects
		/// </summary>
		public event EventHandler MaxDepthChanged;
		#endregion
	}
}
