/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implements a sphere primitive
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments
 *		12/29/2004 - Ben Watson - Added attributes for property editor
 * 
 *
 *--*/
using System;
using System.ComponentModel;
using BenWatson.BRayTracer.Raytracer;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Defines a sphere shape for ray tracing
	/// </summary>
	[Serializable]
	public class Sphere : Shape, ICloneable
	{
		/// <summary>
		/// Radius
		/// </summary>
		protected float m_radius=1.0f;

		#region Temporary variables
		/// <summary>
		/// Square of radius
		/// </summary>
		protected float m_radiusSquared=1.0f;
		#endregion
		
		#region Properties
		/// <summary>
		/// Gets or sets the radius of the sphere
		/// </summary>
		[Viewable, CategoryAttribute("Sphere"), Description("Radius of sphere")]
		public float Radius 
		{
			get 
			{
				return m_radius;
			}
			set 
			{
				m_radius=value;
				if (m_radius<0.0f)
					m_radius=0.0f;
			}
		}
		#endregion


		/// <summary>
		/// Initializes a sphere located at the origin with radius 0.0
		/// </summary>
		public Sphere()
		{
						
		}

		/// <summary>
		/// Initializes a sphere at the origin with the given radius
		/// </summary>
		/// <param name="Radius">Radius of the sphere</param>
		public Sphere(float Radius) 
		{
			m_radius=Radius;
		}

		/// <summary>
		/// Initializes a sphere at the given position with the given radius
		/// </summary>
		/// <param name="Position">Sphere's position</param>
		/// <param name="Radius">Sphere's radius</param>
		public Sphere(Vector3f Position, float Radius)
		{
			this.m_position = Position;
			m_radius=Radius;
		}

		/// <summary>
		/// Returns distance ray must travel to hit sphere
		/// </summary>
		/// <param name="ray">Ray to test</param>
		/// <returns>Returns distance if hit, negative if no hit</returns>
		public override float DistanceToHit(Ray ray)
		{
			//Vector3f rayToSphere  = this.m_position - ray.Start;
			Vector3f rayToSphere2 = m_position;
			//temporarily subtract
			rayToSphere2.Subtract(ray.Start);

			//see if sphere's center is behind ray's starting point
			float distToRayClosest = Vector3f.Dot(rayToSphere2,ray.Direction);
			
			if (distToRayClosest < 0.0f) 
			{
				rayToSphere2.Add(ray.Start);
				return -1.0f;
			}

			//distance from sphere's center to closest point ray comes
			float distSquared = rayToSphere2.MagnitudeSquared - (distToRayClosest * distToRayClosest);
			//fix vector back
			rayToSphere2.Add(ray.Start);

			if (distSquared > m_radiusSquared)
				return -1.0f;

			//must intersect sphere...return distance
			return distToRayClosest - (float)System.Math.Sqrt(m_radiusSquared - distSquared);
		}

		/// <summary>
		/// Calculates the surface normal at a certain point
		/// </summary>
		/// <param name="Point">A point on the surface of the sphere</param>
		/// <returns>Surface normal at the point</returns>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			Vector3f normal = Point - m_position;
			normal.Divide(m_radius);
			return normal;
		}

		/// <summary>
		/// Determines if a ray intersects the sphere
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>True if the rya hits the sphere, false otherwise</returns>
		public override bool IsHit(Ray ray)
		{
			//Vector3f rayToSphere  = this.m_position - ray.Start;
			Vector3f rayToSphere2 = m_position;
			//temporarily subtract
			rayToSphere2.Subtract(ray.Start);

			//see if sphere's center is behind ray's starting point
			float distToRayClosest = Vector3f.Dot(rayToSphere2,ray.Direction);
			
			if (distToRayClosest < 0.0f) 
			{
				rayToSphere2.Add(ray.Start);
				return false;
			}

			//distance from sphere's center to closest point ray comes
			float distSquared = rayToSphere2.MagnitudeSquared - (distToRayClosest * distToRayClosest);
			//fix vector back
			rayToSphere2.Add(ray.Start);

			if (distSquared > m_radiusSquared)
				return false;

			return true;
		}

		/// <summary>
		/// Calculates temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{
			m_radiusSquared = m_radius*m_radius;

		}


		#region ICloneable Members

		/// <summary>
		/// Creates a copy of this sphere
		/// </summary>
		/// <returns>A new sphere object</returns>
		public object Clone()
		{
			Sphere sphere = new Sphere();
			sphere.m_position = (Vector3f)m_position.Clone();
			sphere.m_radius = m_radius;
			sphere.m_material = (Material)m_material.Clone();
			sphere.m_name = (String)m_name.Clone();
			
			return sphere;
		}

		#endregion
	}
}
