/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *		Abstract base class for all shape primitives.
 *
 * Revision History:
 *		12/20/2004 - Added to VSS
 *		12/21/2004 - Added XML comments
 *		12/29/2004 - Added attributes for property editor
 *
 *--*/
using System;
using System.Drawing;
using System.ComponentModel;

using BenWatson.BRayTracer.Raytracer;
namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Represents an abstract shape object
	/// </summary>
	[Serializable]
	public abstract class Shape 
	{
		#region Common Fields
        /// <summary>
        /// Position of shape
        /// </summary>
		protected Vector3f m_position;
		/// <summary>
		/// Bounding sphere
		/// </summary>
		[NonSerialized()]
		protected Sphere m_boundingSphere=null;//cache for efficiency

		/// <summary>
		/// Folder (for organization in tree hierarchy)
		/// </summary>
		protected String m_folder="";
		/// <summary>
		/// Material of object
		/// </summary>
		protected Material m_material;
		/// <summary>
		/// User-defined name of object
		/// </summary>
		protected String m_name="";
		#endregion

		#region Common Properties
		/// <summary>
		/// Gets or sets the folder
		/// </summary>
		[Viewable, CategoryAttribute("General"),ReadOnly(true)]
		public String Folder 
		{
			get 
			{
				return m_folder;
			}
			set 
			{
				m_folder=value;
			}
		}
		/// <summary>
		/// Gets or sets the user-defined name for this object
		/// </summary>
		[Editable, CategoryAttribute("General")]
		public String Name 
		{
			get 
			{
				return m_name;
			}
			set 
			{
				m_name=value;
			}
		}
		/// <summary>
		/// Gets or sets the position for this shape
		/// </summary>
		[Editable, CategoryAttribute("Shape")]
		public virtual Vector3f Position 
		{
			get 
			{
				return m_position;
			}
			set 
			{
				Vector3f diff = value-m_position;
				m_position=value;
				if (m_boundingSphere!=null)
					m_boundingSphere.Position += diff;
			}
		}

		/// <summary>
		/// Gets or sets the material for this shape
		/// </summary>
		[Editable, CategoryAttribute("Shape")]
		public Material Material 
		{
			get 
			{
				return m_material;
			}
			set 
			{
				m_material = value;
			}
		}
		#endregion

		/// <summary>
		/// Initializes basic shape properties with default values
		/// </summary>
		protected Shape() 
		{
			m_position = new Vector3f(0.0f,0.0f,0.0f);
			m_material = new Material();
         }

		/// <summary>
		/// Gets the material at a specific point
		/// </summary>
		/// <param name="Point">The point to check</param>
		/// <returns>A material</returns>
		public virtual Material GetMaterialAtPoint(Vector3f Point) 
		{
			return this.m_material;
		}

		/// <summary>
		/// Calculates the distance from a ray to a shape
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>Distance from ray to shape, or negative if no intersection</returns>
		public abstract float DistanceToHit(Ray ray);
		/// <summary>
		/// Determines if ray hits shape
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>True if ray intersects shape, false otherwise</returns>
		public abstract bool IsHit(Ray ray);
		/// <summary>
		/// Gets the surface normal at a certain point
		/// </summary>
		/// <param name="Point">The point to check</param>
		/// <returns>Surface normal at the point</returns>
		/// <remarks>This procedure assumes that the point is on the shape.</remarks>
		public abstract Vector3f NormalAtPoint(Vector3f Point);
		/// <summary>
		/// Calculates temporary variables for use in rendering
		/// </summary>
		public abstract void DoPreRenderCalculations();
				
		
	}
}
