/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implements a polygon mesh.
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments
 *		12/29/2004 - Ben Watson - Added attributes for property editor
 *
 *--*/
using System;
using System.IO;
using System.ComponentModel;

using BenWatson.BRayTracer.Raytracer;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Represents a polygonal mesh object
	/// </summary>
	[Serializable]
	public class PolygonMesh : Shape
	{
		/// <summary>
		/// Polygon list
		/// </summary>
		private Polygon[] m_polygons = null;
		/// <summary>
		/// Vertex list
		/// </summary>
        private Vector3f[] m_vertices = null;

		#region temporary variables
		/// <summary>
		/// temporary variable to hold closest polygon
		/// </summary>
		/// <remarks>This is used to store the closest polygon between calls to
		/// DistanceToHit() and NormalAtPoint to avoid recalculating the nearest polygon.</remarks>
		private Polygon m_closestPolygon = null;
		#endregion

		/// <summary>
		/// File where mesh was loaded from
		/// </summary>
		private String m_originalFilename = null;


		#region Public Properties
		/// <summary>
		/// Gets the filename of the file where mesh was loaded from
		/// </summary>
		[Category("Polygon Mesh"), Description("Where the mesh was loaded from")] 
		public String OriginalFilename 
		{
			get 
			{
				return m_originalFilename;
			}
		}
		/// <summary>
		/// Gets the number of polygons in mesh
		/// </summary>
		[Category("Polygon Mesh"),Description("Number of polygons in the mesh")]
		public int NumPolygons 
		{
			get 
			{
				if (m_polygons==null)
					return 0;
				return m_polygons.Length;
			}
		}
		/// <summary>
		/// Gets the number of vertices in mesh
		/// </summary>
		/// 
		[Category("Polygon Mesh"),Description("Number of vertices in the mesh")]
		public int NumVertices 
		{
			get 
			{
				if (m_vertices==null)
					return 0;
				return m_vertices.Length;
			}
		}
		#endregion
		
		/// <summary>
		/// Initializes an empty polygon mesh
		/// </summary>
		public PolygonMesh()
		{
			
		}

		/// <summary>
		/// Loads a mesh from a .d file
		/// </summary>
		/// <param name="filename">Filename of mesh</param>
		/// <returns>True if success, false otherwise</returns>
		public bool LoadFromFile(String filename) 
		{
			float minX=float.MaxValue, maxX=float.MinValue, 
				minY=float.MaxValue, maxY=float.MinValue, 
				minZ=float.MaxValue, maxZ=float.MinValue;

			FileStream stream = File.OpenRead(filename);
			if (stream==null)
				return false;

            StreamReader reader = new StreamReader(stream);

			//get number of vertices and polygons
			String line = reader.ReadLine();
			String[] tokens = line.Split(' ','\t');
            tokens = RemoveEmpty(tokens);
			
			int numVertices = 0;
			int numPolygons = 0;

			if (tokens.Length==2) 
			{
				numVertices=int.Parse(tokens[0]);
				numPolygons=int.Parse(tokens[1]);
			} 
			else if (tokens.Length==3) 
			{
				numVertices=int.Parse(tokens[1]);
				numPolygons=int.Parse(tokens[2]);
			} else 
				return false;

			try 
			{
				//read in vertices
				m_vertices = new Vector3f[numVertices];
				for (int i=0;i<numVertices;i++) 
				{
					line = reader.ReadLine();
					tokens = line.Split(' ','\t');
					tokens = RemoveEmpty(tokens);

					Vector3f vertex = new Vector3f();
					vertex.X = float.Parse(tokens[0]);
					vertex.Y = float.Parse(tokens[1]);
					vertex.Z = float.Parse(tokens[2]);
					if (vertex.X<minX) minX=vertex.X;
					else if (vertex.X>maxX) maxX=vertex.X;
					if (vertex.Y<minY) minY=vertex.Y;
					else if (vertex.Y>maxY) maxY=vertex.Y;
					if (vertex.Z<minZ) minZ=vertex.Z;
					else if (vertex.Z>maxZ) maxZ=vertex.Z;
					
					m_vertices[i] = vertex;
				}

				//read in polygons
				m_polygons = new Polygon[numPolygons];
				for (int i=0;i<numPolygons;i++) 
				{
					line = reader.ReadLine();
					tokens = line.Split(' ','\t');
					tokens = RemoveEmpty(tokens);
					int n = int.Parse(tokens[0]);
					Vector3f[] polyVertices = new Vector3f[n];
					for (int j=0, k=tokens.Length-1;j<n && k>0;j++,k--) 
					{
						//reverse order--all of our polygons must be in counterclockwise order
						polyVertices[j] = m_vertices[int.Parse(tokens[k])-1];
                    }
					Polygon polygon = new Polygon(polyVertices);					
					m_polygons[i] = polygon;
				}
				
				Vector3f spherePos = new Vector3f((minX+maxX)/2,(minY+maxY)/2,(minZ+maxY)/2);

				float width = Math.Abs(maxX-minX);
				float height = Math.Abs(maxY-minY);
				float length = Math.Abs(maxZ-minZ);
				float radius = Math.Max(width,Math.Max(height,length))/2;

				this.m_boundingSphere = new Sphere(spherePos,radius);
				
				m_originalFilename = filename;

				return true;
			}
			catch (Exception ex) 
			{
				System.Diagnostics.Debug.WriteLine(ex.Message);
				return false;
			}
			finally 
			{
				reader.Close();
			}
                   
		}

		/// <summary>
		/// Removes empty or null strings from an array
		/// </summary>
		/// <param name="strs">String array to process</param>
		/// <returns>a new string array with empty nodes removed</returns>
		private String[] RemoveEmpty(String[] strs) 
		{
			int count = 0;
			for (int i=0;i<strs.Length;i++)
				if (strs[i]!=null && strs[i].Length>0)
					count++;
			String[] tokens = new string[count];
			for (int i=0,j=0;i<strs.Length;i++)
				if (strs[i]!=null && strs[i].Length>0) 
				{
					tokens[j]=strs[i];
					j++;
				}
			return tokens;
			
		}


		/// <summary>
		/// Calculates temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{
			foreach(Polygon P in this.m_polygons) 
			{
				P.DoPreRenderCalculations();
			}

		}

		/// <summary>
		/// Returns the distance from the ray's origin to the nearest polygon
		/// in the mesh
		/// </summary>
		/// <param name="ray">Ray</param>
		/// <returns>Distance from ray's origin to a polygon in this mesh, or negative if no intersection</returns>
		public override float DistanceToHit(Ray ray)
		{
			if (this.m_boundingSphere.DistanceToHit(ray)<0.0f)
				return -1.0f;

			Polygon closestPolygon = null;
			float closestDistance = float.MaxValue;
			for (int i=0;i<m_polygons.Length;i++) 
			{
                Polygon poly = m_polygons[i];
				float dist = poly.DistanceToHit(ray);
				if (dist>0.0f && dist < closestDistance) 
				{
					closestDistance = dist;
					closestPolygon = poly;
				}
			}

			if (closestPolygon!=null) 
			{
				this.m_closestPolygon = closestPolygon;
				return closestDistance;
			}
			return -1.0f;
		}

		/// <summary>
		/// Determines if a ray hits the mesh
		/// </summary>
		/// <param name="ray">The ray to test</param>
		/// <returns>True if ray hits the mesh, false otherwise</returns>
		public override bool IsHit(Ray ray)
		{
			if (this.m_boundingSphere.IsHit(ray)==false)
				return false;

			
			for (int i=0;i<m_polygons.Length;i++) 
			{
				if (m_polygons[i].IsHit(ray)==true)
					return true;
			}
			return false;
		}

		/// <summary>
		/// Returns the surface normal for the intersected polygon
		/// </summary>
		/// <param name="Point">The point</param>
		/// <returns>Surface normal for intersected polygon</returns>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			//uses the existing m_closestPolygon as the polygon
			//for efficiency.
			//m_closestPolygon is set in DistanceToHit
			if (m_closestPolygon!=null)
			{
				return m_closestPolygon.NormalAtPoint(Point);
			}
			return null;
		}

	}
}
