/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implementation class for a polygon primitive
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments
 *		12/29/2004 - Ben Watson - Added attributes for property editor
 *		12/29/2004 - Ben Watson - Changed m_vertices to be a dynamic ArrayList. Converted to an array in pre-render.
 *
 *--*/

using System;
using System.Collections;
using System.ComponentModel;

using BenWatson.BRayTracer.Raytracer;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Describes a polygon primitive
	/// </summary>
	/// <remarks>The polygon must be convex.</remarks>
	[Serializable]
	public class Polygon : Shape, ICloneable
	{
		/// <summary>
		/// List of vertices
		/// </summary>
		//private Vector3f[] m_vertices = null;
		private System.Collections.ArrayList m_vertices = new System.Collections.ArrayList();
		/// <summary>
		/// Surface normal for polygon.
		/// </summary>
		private Vector3f m_normal = null;
		
		#region Temporary variables

		[NonSerialized()]
		private float m_distance=0.0f;
		[NonSerialized()]
		private Vector3f[] m_tempVertices = null;
		[NonSerialized()]
		private static float MATCH_FACTOR = 0.999f;
		[NonSerialized()]
		private static float m_maxAngle=(MATCH_FACTOR * (2.0f * (float)Math.PI));
        //private bool m_bClockwise = false;
		#endregion

		#region Public Properties
		/// <summary>
		/// Gets or sets the set of vertices used by this polygon
		/// </summary>
		[Category("Polygon"), Description("List of vertices in this polygon")]
		/*
		public Vector3f[] Vertices 
		{
			get 
			{
				return m_vertices;
			}
			set 
			{
				m_vertices=value;
				m_position=value[0];
			}
		}
		*/
		public ArrayList Vertices 
		{
			get 
			{
				return m_vertices;
			}
			set 
			{
				m_vertices=value;
				if (m_vertices!=null && m_vertices.Count >0) 
				{
					m_position=(Vector3f)m_vertices[0];
				}
			}

		}
		/// <summary>
		/// Gets the number of vertices in this polygon
		/// </summary>
		[Category("Polygon"),Description("Number of vertices in polygon")]
		public int NumVertices 
		{
			get 
			{
				if (m_vertices==null)
					return 0;
				//return m_vertices.Length;
				return m_vertices.Count;
			}
		}

		/// <summary>
		/// Determines if the polygon is planar
		/// </summary>
		[Category("Polygon"), Description("True if polygon lies in a plane, false otherwise")]
		public bool IsPlanar 
		{
			get 
			{
				if (m_vertices.Count<=3)
					return true;

				Vector3f normal = Vector3f.Normal(m_vertices[0] as Vector3f,m_vertices[1] as Vector3f,m_vertices[2] as Vector3f);

				for (int i=1;i<m_vertices.Count-2;i++) 
				{
                    Vector3f n = Vector3f.Normal(m_vertices[i] as Vector3f,m_vertices[i+1] as Vector3f,m_vertices[i+2] as Vector3f);
                    if (n!=normal)
						return false;
				}
				return true;
			}
		}

		
		#endregion

		/// <summary>
		/// Initializes an empty polygon
		/// </summary>
		public Polygon()
		{
		}

		/// <summary>
		/// Initializes a polygon from a given set of vertices. Assumes clockwise vertices.
		/// </summary>
		/// <param name="Vertices">Vertices that will make up this polygon</param>
		public Polygon(Vector3f[] Vertices) 
		{
			this.m_vertices = new ArrayList(Vertices);
			CalcNormal();

			m_position = (Vector3f)m_vertices[0];
			m_distance = Vector3f.Dot(m_position, m_normal);
		}

		/// <summary>
		/// Calculates temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{
			CalcNormal();
			m_tempVertices = (Vector3f[])this.m_vertices.ToArray(typeof(Vector3f));
			m_distance = Vector3f.Dot(m_position, m_normal);
		}

		/// <summary>
		/// Calculates the distance from a ray to the polygon.
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>The distance from the ray to the polygon, or negative if no intersection</returns>
		public override float DistanceToHit(Ray ray)
		{
			float dist = (Vector3f.Dot(ray.Start, m_normal) -  m_distance) /
				-Vector3f.Dot(m_normal,ray.Direction);
		
			if (dist<0.0) return -1.0f;
			//Vector3f pt = ray.Start + (ray.Direction * dist);
			//temporarily mangle to avoid creating a new one
			ray.Direction.Multiply(dist);
			ray.Direction.Add(ray.Start);

			if (PointIsInPolygon(ray.Direction)) 
			{
				ray.Direction.Subtract(ray.Start);
				ray.Direction.Divide(dist);

				return dist;
			}
			ray.Direction.Subtract(ray.Start);
			ray.Direction.Divide(dist);

			return -1.0f;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="Point"></param>
		/// <returns></returns>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			return m_normal;
		}

		/// <summary>
		/// Determines if a ray hits the polygon.
		/// </summary>
		/// <param name="ray">The ray to trace.</param>
		/// <returns>True if ray hits polygon, false otherwise.</returns>
		public override bool IsHit(Ray ray)
		{
			return (DistanceToHit(ray)>=0.0f);
		}

		/// <summary>
		/// Calculates the surface normal using the first three vertices of the polygon
		/// </summary>
		private void CalcNormal() 
		{
			m_normal = Vector3f.Normal(m_vertices[0] as Vector3f,m_vertices[1] as Vector3f,m_vertices[2] as Vector3f);
		}

		/// <summary>
		/// Determines if a point is in the polygon
		/// </summary>
		/// <param name="point">The point to test</param>
		/// <returns>True if point is inside, false otherwise</returns>
		private bool PointIsInPolygon(Vector3f point) 
		{
			double angle=0.0;
			Vector3f a,b;
			
			/*for (int i=0;i<m_vertices.Length;i++) 
			{
				a = m_vertices[i]-point;
				b = m_vertices[(i+1)%m_vertices.Length]-point;

				angle+=Vector3f.AngleRadians(a,b);
			}*/
			a = m_tempVertices[0];
			a.Subtract(point);
			b = a;
			for (int i=0;i<m_tempVertices.Length;i++) 
			{
				b = m_tempVertices[(i+1)%m_tempVertices.Length];
				b.Subtract(point);

				angle+=Vector3f.AngleRadians(a,b);
				a.Add(point);

				a=b;
			}
			b.Add(point);
			
			if (angle>= m_maxAngle)
				return true;
			return false;
		}
		#region ICloneable Members

		/// <summary>
		/// Creates a copy of this polygon
		/// </summary>
		/// <returns>Copy of the polygon</returns>
		public object Clone()
		{
			Polygon p = new Polygon();
			if (NumVertices>0) 
			{
				/*
				Vector3f[] pv = new Vector3f[NumVertices];
				for (int i=0;i<pv.Length;i++)
					pv[i] = this.m_vertices[i];
				p.Vertices = pv;
				p.CalcNormal();
				*/

				ArrayList pv = new ArrayList();

				int num = NumVertices;
				for (int i=0;i<num;i++) 
				{
					pv.Add(((Vector3f)m_vertices[i]).Clone());
				}
				p.Vertices = pv;
			}
			return p;
		}

		#endregion
	}
}
