/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *		Implementation class for a plane primitive  
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments 
 *		12/21/2004 - Ben Watson - removed superfluous Position property override
 *		12/29/2004 - Ben Watson - Added attributes for property editor
 *
 *--*/
using System;
using System.ComponentModel;

using BenWatson.BRayTracer.Raytracer;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Implements a Plane primitive
	/// </summary>
	[Serializable]
	public class Plane : Shape, ICloneable
	{
		/// <summary>
		/// Surface normal of plane
		/// </summary>
		private Vector3f m_normal;

		#region Temporary variables
		[NonSerialized()]
		private float m_distance=0.0f;
		#endregion

		#region Properties
		/// <summary>
		/// Gets or sets the surface normal of the plane
		/// </summary>
		[Editable, Category("Plane"),Description("Surface normal of plane")]
		public Vector3f Normal 
		{
			get 
			{
				return m_normal;
			}
			set 
			{
				m_normal=value;
				m_normal.Normalize();
			}
		}

		/// <summary>
		/// Gets or sets the origin of the surface normal for the plane
		/// </summary>
		[Editable, Category("Plane"), Description("Origin of plane")]
		public Vector3f Origin 
		{
			get 
			{
				return m_position;
			}
			set 
			{
				m_position=value;
				//TODO: note this won't update if we change a vector's values individually!
				m_distance = Vector3f.Dot(m_position, m_normal);
			}
		}

	
		#endregion

		/// <summary>
		/// Initializes a plane on the XZ-plane
		/// </summary>
		public Plane()
		{
			m_normal = new Vector3f(0.0f,1.0f,0.0f);
			this.m_position = new Vector3f(0.0f,0.0f,0.0f);
			
		}

		/// <summary>
		/// Initializes a new plane from the given origin and normal data
		/// </summary>
		/// <param name="Origin">Origin of normal</param>
		/// <param name="Normal">Surface normal of plane</param>
		public Plane(Vector3f Origin, Vector3f Normal) 
		{
			this.m_position = Origin;
			this.m_normal=Normal;
			
		}

		/// <summary>
		/// Calculates temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{
			m_normal.Normalize();
			m_distance = Vector3f.Dot(m_position, m_normal);
		}


		/// <summary>
		/// Calculates distance from ray to plane
		/// </summary>
		/// <param name="ray">Ray to trace</param>
		/// <returns>Distance from ray to plane, or negative if ray does not intersect plane.</returns>
		public override float DistanceToHit(Ray ray)
		{
			return (Vector3f.Dot(ray.Start, m_normal) -  m_distance) /
				-Vector3f.Dot(m_normal,ray.Direction);
		}

		/// <summary>
		/// Determines if a ray hits the plane
		/// </summary>
		/// <param name="ray">Ray to optimize</param>
		/// <returns>True if ray intersects plane, false if it doesn't</returns>
		public override bool IsHit(Ray ray)
		{
			//TODO: can optimize this (don't need denominator)
			return DistanceToHit(ray) >=0.0f;
		}


		/// <summary>
		/// Returns the surface normal for the plane
		/// </summary>
		/// <param name="Point">Point</param>
		/// <returns>surface normal</returns>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			return m_normal;
		}

		
		#region ICloneable Members

		/// <summary>
		/// Makes a copy of this plane
		/// </summary>
		/// <returns></returns>
		public object Clone()
		{
			Plane plane = new Plane();
			plane.m_position = (Vector3f)this.m_position.Clone();
			plane.m_normal = (Vector3f)this.m_normal.Clone();
			plane.m_material = (Material)this.m_material.Clone();
			plane.m_distance = this.m_distance;
			//plane.m_folder = (String)this.m_folder.Clone();
			plane.m_name = (String)this.m_name.Clone();

			return plane;
		}

		#endregion
	}
}
