/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *    Implementation class for Disc primitive 
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments
 *		12/29/2004 - Ben Watons - Added attributes for property editors
 *
 *--*/

using System;
using System.ComponentModel;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Implements a one-sided disc primitive.
	/// </summary>
	public class Disc : Shape
	{
		/// <summary>
		/// Radius
		/// </summary>
		private float m_radius = 1.0f;
		/// <summary>
		/// Normal of disc (one-sided)
		/// </summary>
		private Vector3f m_normal =new Vector3f(0.0f,1.0f,0.0f);
		
		#region temporary variables
		[NonSerialized()]
		private float m_distance = 0.0f;
		[NonSerialized()]
		private float m_radiusSquared=1.0f;
		#endregion

		#region Properties
		/// <summary>
		/// Gets or sets the radius of the disc
		/// </summary>
		[Viewable, Category("Disc"), Description("Radius of disc")]
		public float Radius 
		{
			get 
			{
				return m_radius;
			}
			set 
			{
				m_radius=value;
				if (m_radius<0.0f)
					m_radius=0.0f;
			}
		}

		/// <summary>
		/// Gets or sets the surface normal for the disc
		/// </summary>
		[Editable, Category("Disc"), Description("Surface normal for disc")]
		public Vector3f Normal 
		{
			get 
			{
				return m_normal;
			}
			set 
			{
				m_normal = value;
				m_normal.Normalize();
			}
		}
		#endregion


		/// <summary>
		/// Initializes a new disc at the origin
		/// </summary>
		public Disc()
		{
			m_position = new Vector3f(0.0f,0.0f,0.0f);
		}

		/// <summary>
		/// Initializes a new disc
		/// </summary>
		/// <param name="Position">Position of the disc</param>
		/// <param name="Normal">Surface normal of the disc</param>
		/// <param name="Radius">Radius of the disc</param>
		public Disc(Vector3f Position, Vector3f Normal, float Radius) 
		{
			m_position = Position;
			m_normal = Normal;
			m_radius=Radius;
		}

		/// <summary>
		/// Calculate temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{
			m_normal.Normalize();
			m_distance = Vector3f.Dot(m_position,m_normal);
			m_radiusSquared = m_radius*m_radius;

		}

		/// <summary>
		/// Calculates distance from a ray to this shape.
		/// </summary>
		/// <param name="ray">Ray to trace</param>
		/// <returns>Distance from ray to disc, or negative value if no intersection</returns>
		public override float DistanceToHit(BenWatson.BRayTracer.Raytracer.Ray ray)
		{
			//see if ray hits plane

			float dist = (Vector3f.Dot(ray.Start, m_normal) -  m_distance) /
				-Vector3f.Dot(m_normal,ray.Direction);

			if (dist<0.0f)
				return -1.0f;

			//modify vector in place instead of copying
			Vector3f pt = ray.Direction;
			pt.Multiply(dist);
			pt.Add(ray.Start);
			pt.Subtract(m_position);

			float magSquared = pt.MagnitudeSquared;
			//fix vector back
			pt.Add(m_position);
			pt.Subtract(ray.Start);
			pt.Divide(dist);

			if (magSquared<=m_radiusSquared)
				return dist;
			return -1.0f;
		}

		/// <summary>
		/// Returns surface normal at the given point
		/// </summary>
		/// <param name="Point">Point at which to evaluate normal</param>
		/// <returns>Surface normal</returns>
		/// <remarks>The surface normal is the same everywhere because the disc is planar</remarks>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			return m_normal;
		}

		/// <summary>
		/// Determines if a ray hits the disc
		/// </summary>
		/// <param name="ray">Ray to trace</param>
		/// <returns>True if ray hits disc, false otherwise</returns>
		public override bool IsHit(BenWatson.BRayTracer.Raytracer.Ray ray)
		{
			//see if ray hits plane

			float dist = (Vector3f.Dot(ray.Start, m_normal) -  m_distance) /
				-Vector3f.Dot(m_normal,ray.Direction);

			if (dist<0.0f)
				return false;

			//modify vector in place instead of copying
			Vector3f pt = ray.Direction;
			pt.Multiply(dist);
			pt.Add(ray.Start);
			pt.Subtract(m_position);

			float magSquared = pt.MagnitudeSquared;
			//fix vector back
			pt.Add(m_position);
			pt.Subtract(ray.Start);
			pt.Divide(dist);

			if (magSquared<=m_radiusSquared)
				return true;
			return false;
		}




	}
}
