/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implementation of a cylinder primitive
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments
 *		12/29/2004 - Ben Watson - Added attributes for property editor.
 *		12/29/2004 - Ben Watson - Fixed - force radius/height to be >= 0.0
 *		12/29/2004 - Ben Watson - Added ICloneable
 * 
 *--*/
using System;
using System.ComponentModel;

namespace BenWatson.BRayTracer.Primitives
{
	
	/// <summary>
	/// Implementation for a cylinder primitive.
	/// </summary>
	/// <remarks>Cylinders can be capped or uncapped. If they are uncapped, then the inside surface is visible.</remarks>
	[Serializable]
	public class Cylinder: Shape, ICloneable
	{
		/// <summary>
		/// Height
		/// </summary>
		private float m_height=1.0f;
		/// <summary>
		/// Radius
		/// </summary>
		private float m_radius=1.0f;
		/// <summary>
		/// Axis direction
		/// </summary>
		/// <remarks>Default is along positive Y axis (straight up)</remarks>
		private Vector3f m_axis = new Vector3f(0.0f,1.0f,0.0f);

		/// <summary>
		/// Cylinder is capped
		/// </summary>
		private bool m_capped=false;

		#region temporary instance variables
		[NonSerialized()]
		private float m_tolerance=0.001f;
		[NonSerialized()]
		private float m_radiusSquared = 1.0f;
		[NonSerialized()]
		private float m_topDist = 0.0f;
		[NonSerialized()]
		private float m_bottomDist = 0.0f;
		[NonSerialized()]
		private float m_distanceTop=0.0f;
		[NonSerialized()]
		private float m_distanceBottom=0.0f;
		[NonSerialized()]
		private Vector3f m_topPos;
		[NonSerialized()]
		private Vector3f m_inverseAxis;
		[NonSerialized()]
		private Disc m_topCap;
		[NonSerialized()]
		private Disc m_baseCap;
		[NonSerialized()]
		private IntersectionArea m_intersect = IntersectionArea.None;
		#endregion

		private enum IntersectionArea {OuterSide, InnerSide, Base, Cap, None};

		#region Properties

		/// <summary>
		/// Gets or sets the axis direction
		/// </summary>
		[Editable, Category("Cylinder"), Description("Unit axis vector")]
		public Vector3f Axis 
		{
			get 
			{
				return m_axis;
			}
			set 
			{
				m_axis = value;
				m_axis.Normalize();
			}
		}
		/// <summary>
		/// Gets or sets the cylinder height
		/// </summary>
		[Editable, Category("Cylinder"), Description("Height of cylinder from base")]
		public float Height 
		{
			get 
			{
				return m_height;
			}
			set 
			{
				m_height=value;
				if (m_height<0.0f)
					m_height=0.0f;
			}
		}

		/// <summary>
		/// Gets or sets the cylinder radius
		/// </summary>
		[Editable, Category("Cylinder"), Description("Radius of cylinder.")]
		public float Radius 
		{
			get 
			{
				return m_radius;
			}
			set 
			{
				m_radius=value;
				if (m_radius<0.0f)
					m_radius=0.0f;
			}
		}

		/// <summary>
		/// Gets or sets the capped state
		/// </summary>
		[Editable, Category("Cylinder"), Description("Specifies whether the cylinder has caps on the ends")]
		public bool Capped 
		{
			get 
			{
				return m_capped;
			}
			set 
			{
				m_capped=value;
			}
		}
		#endregion
	
		/// <summary>
		/// Initializes a default vertical cylinder at the origin with radius 1.0, height 1.0
		/// </summary>
		public Cylinder() 
		{

		}
		/// <summary>
		/// Initializes a cylinder
		/// </summary>
		/// <param name="Position">Position of bottom of the cylinder</param>
		/// <param name="Height">Height of cylinder</param>
		/// <param name="Radius">Radius of cylinder</param>
		public Cylinder(Vector3f Position, float Height, float Radius)
		{
			this.m_position = Position;
			this.m_height = Height;
			this.m_radius = Radius;
		}

		/// <summary>
		/// Initializes a cylinder
		/// </summary>
		/// <param name="Position">Position of bottom of the cylinder</param>
		/// <param name="Height">Height of cylinder</param>
		/// <param name="Radius">Radius of cylinder</param>
		/// <param name="Axis">Axis direction of cylinder</param>
		public Cylinder(Vector3f Position, Vector3f Axis, float Height, float Radius)
		{
			this.m_position = Position;
			this.m_axis = Axis;
			this.m_height = Height;
			this.m_radius = Radius;
		}
			
		/// <summary>
		/// Calculate a number of temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{
			m_axis.Normalize();
			m_inverseAxis =m_axis * -1.0f;

			m_topPos = m_axis * m_height ;
			m_topPos.Add(m_position);

			m_distanceTop = Vector3f.Dot(m_topPos, m_axis);
			m_distanceBottom = Vector3f.Dot(m_position,m_axis * -1);

			m_topDist = Vector3f.Dot(m_topPos,m_axis);

			m_radiusSquared = m_radius * m_radius;

			Vector3f midPoint = (m_axis * (m_height/2));
			midPoint.Add(m_position);

			float sphereRadius = Math.Max(m_height/2, m_radius)*1.05f;
			m_boundingSphere = new Sphere(midPoint, sphereRadius);
			
			m_boundingSphere.DoPreRenderCalculations();

			if (m_capped) 
			{
				m_topCap = new Disc(m_topPos,m_axis,m_radius*1.01f);
				m_topCap.Material = this.m_material;

				m_baseCap = new Disc(m_position,m_inverseAxis,m_radius*1.01f);
				m_baseCap.Material = this.m_material;
			}
		}

		/// <summary>
		/// Calculate the distance from a ray to the cylinder along the ray's direction.
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>Distance from ray to cylinder, or less than 0 if ray doesn't hit cylinder</returns>
		/// <remarks>If the ray hits the near side of the cylinder, the distance to that is returned.
		/// If it hits an end cap, and the cylinder is capped, it returns the distance to the cap.
		/// If it hits an end cap, and the cylinder is not capped, it returns the distance to the other side of the cylinder.
		/// Otherwise, the ray didn't hit the cylinder at all and it returns a negative number.</remarks>
		public override float DistanceToHit(BenWatson.BRayTracer.Raytracer.Ray ray)
		{
			if (m_boundingSphere.DistanceToHit(ray)<0.0f) return -1.0f;

			float dist=-1.0f, sideDist=-1.0f, detSqrRt=0.0f;
			
			
			Vector3f pq = (Vector3f)ray.Start.Clone();
			pq.Subtract(m_position);

			float alpha = Vector3f.Dot(m_axis,ray.Direction);
			float beta = Vector3f.Dot(m_axis, pq);
			float gamma = Vector3f.Dot(pq,ray.Direction);

			float b = 2*(gamma-beta*alpha);
			float a = 1.0f - alpha*alpha;

			float det = b*b - 4*a*(pq.MagnitudeSquared - beta*beta - m_radiusSquared);

			if (det-m_tolerance<0.0f) 
			{
				//if ray doesn't intersect infinite cylinder, we can just return here
				return -1.0f;
			}
			else 
			{
				detSqrRt = (float)Math.Sqrt(det);
				sideDist = (-b - detSqrRt)/(2*a);
			}
			
			
			//it hit infinite cylinder, so check if it hit finite cylinder

		
			//check to see if it hit finite cylinder
            
			IntersectionArea sideIntersectArea = IntersectionArea.None;
			//get intersection point
			Vector3f point = ray.Direction * (sideDist * 0.999f);
			point.Add(ray.Start);
			point.Subtract(m_position);
			float ptHeight = Vector3f.Dot(m_axis,point);
			if (ptHeight>=-m_tolerance && ptHeight<=m_height+m_tolerance) 
			{
				sideIntersectArea = IntersectionArea.OuterSide;
				//return sideDist;
			}
			else if (!m_capped) 
			{
				//check other side of cylinder
				float otherDist = (-b + detSqrRt)/(a+a);
				point = ray.Direction * (otherDist * 0.999f);
				point.Add(ray.Start);
				point.Subtract(m_position);
				ptHeight = Vector3f.Dot(m_axis,point);
				if (ptHeight>=0.0f && ptHeight<=m_height) 
				{
					sideDist= otherDist;
					sideIntersectArea= IntersectionArea.InnerSide;

				}
				else 
				{
					sideDist= -1.0f;
					sideIntersectArea = IntersectionArea.None;
				}
			} 
			else 
			{
				sideDist=-1.0f;
				sideIntersectArea = IntersectionArea.None;
			}

			if (m_capped) 
			{

				//top cap
				/*m_topDist = (Vector3f.Dot(ray.Start, m_axis) -  m_distanceTop) /
					-Vector3f.Dot(m_axis,ray.Direction);

				if (m_topDist>=0.0f) 
				{
					Vector3f pt = ray.Direction * m_topDist;
					pt.Add(ray.Start);
					pt.Subtract(m_topPos);
					
					if (pt.MagnitudeSquared> m_radiusSquared)
						m_topDist=-1.0f;
				}

				dist = m_topDist;
				*/
				dist = m_topCap.DistanceToHit(ray);
				m_intersect = IntersectionArea.Cap;

				//bottom cap

				//invert axis
				/*
				m_axis.Multiply(-1.0f);

				m_bottomDist = (Vector3f.Dot(ray.Start, m_axis) -  m_distanceBottom) /
					-Vector3f.Dot(m_axis,ray.Direction);

				if (m_bottomDist>=0.0f) 
				{
					Vector3f pt = ray.Direction * m_bottomDist;
					pt.Add(ray.Start);
					pt.Subtract(m_position);

					if (pt.MagnitudeSquared > m_radiusSquared)
						m_bottomDist = -1.0f;
				}
				//fix axis
				m_axis.Multiply(-1.0f);
				*/

				m_bottomDist = m_baseCap.DistanceToHit(ray);
				
				//pick closest one
				/*if (m_topDist < 0.0f || (m_bottomDist < m_topDist && m_bottomDist>0.0f))
					dist = m_bottomDist;
					*/
				if (dist < 0.0f || (m_bottomDist < dist && m_bottomDist>0.0f)) 
				{
					dist = m_bottomDist;
					m_intersect = IntersectionArea.Base;

				}
			
			}

			if (dist<0.0f || (sideDist < dist && sideDist>0.0f)) 
			{
				this.m_intersect = sideIntersectArea;
				return sideDist;
			}
        
			return dist;
		}

		/// <summary>
		/// Determines if the ray hits the cylinder or not.
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>True if the ray intersects the cylinder, false otherwise.</returns>
		public override bool IsHit(BenWatson.BRayTracer.Raytracer.Ray ray)
		{
			if (m_boundingSphere.IsHit(ray)==false) return false;

			return (DistanceToHit(ray)>=0.0f);
		}

		/// <summary>
		/// Returns the surface normal at the given point.
		/// </summary>
		/// <param name="Point">Point on the surface of the object</param>
		/// <returns>A normal vector at the given point on the surface</returns>
		/// <remarks>Assumes that Point is on the surface of this object.</remarks>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			/*Vector3f normal = (Vector3f)Point.Clone();
			normal.Subtract(m_position);

			
			if (m_capped) 
			{
				float ptHeight = Vector3f.Dot(m_axis,normal);
				if (ptHeight>=m_height)
					return m_axis;
				else if (ptHeight<=0.0f && ptHeight + m_tolerance>0.0f)
					return m_inverseAxis;
			} 
			Vector3f temp = m_axis * Vector3f.Dot(m_axis,normal);
			normal.Subtract(temp);
			normal.Divide(m_radius);
			if (!m_capped) {	
				
				//not capped, so determine if we're inside or outside
				//if inside, we need to invert the normal
                Vector3f p  = Point;// - m_position;
				p.Subtract(m_position);
				if (p.Magnitude < m_radius) 
					normal.Multiply(-1.0f);
				p.Add(m_position);
                
			}
			return normal;
			*/
			
			Vector3f normal=null;

			if (m_intersect==IntersectionArea.OuterSide) 
			{
				normal = (Vector3f)Point.Clone();
				normal.Subtract(m_position);

				float d = Vector3f.Dot(m_axis,normal);
				m_axis.Multiply(d);
				normal.Subtract(m_axis);
				m_axis.Divide(d);

				normal.Divide(m_radius);
			}
			else if (m_intersect==IntersectionArea.InnerSide) 
			{
				normal = (Vector3f)Point.Clone();
				normal.Subtract(m_position);

				Vector3f temp = m_axis * Vector3f.Dot(m_axis,normal);
				float d = Vector3f.Dot(m_axis,normal);
				m_axis.Multiply(d);
				normal.Subtract(m_axis);
				m_axis.Divide(d);

				normal.Divide(-m_radius);
			}
			else if (m_intersect==IntersectionArea.Cap) 
			{
                normal = this.m_topCap.Normal;
			}
			else if (m_intersect==IntersectionArea.Base) 
			{
				normal = this.m_baseCap.Normal;
			}

			return normal;
		
		}
		#region ICloneable Members

		/// <summary>
		/// Creates a copy of this cylinder
		/// </summary>
		/// <returns>A copy of the cylinder</returns>
		public object Clone()
		{
			Cylinder c = new Cylinder();
			c.Capped = Capped;
			c.Axis = (Vector3f)this.Axis.Clone();
			c.Position = (Vector3f)this.Position.Clone();
			c.Radius  = Radius;
			c.Height = Height;
			//TODO: fix this line
			//c.Material = (Material)this.m_material.Clone();
			c.Material = Material;
			c.Name = Name;
			c.Folder = Folder;

			return c;
		}

		#endregion
	}
}
