/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implementation of a cone primitive
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 * 
 *
 *--*/
using System;
using BenWatson.BRayTracer;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// Implementation for a cone primitive - INCOMPLETE
	/// </summary>
	public class Cone : Shape
	{
		private Vector3f m_baseNormal=new Vector3f(0.0f,1.0f,0.0f);
		private float m_height=2.0f;
		private float m_radius=1.0f;


		/// <summary>
		/// Gets or sets the height of the cone
		/// </summary>
		public float Height 
		{
			get 
			{
				return m_height;
			}
			set 
			{
				m_height=value;
			}
		}

		/// <summary>
		/// Gets or sets the normal for the base circle
		/// </summary>
		public Vector3f Up 
		{
			get 
			{
				return m_baseNormal;
			}
			set 
			{
				m_baseNormal = value;
				m_baseNormal.Normalize();
			}
		}

		/// <summary>
		/// Gets or sets the radius of the cone.
		/// </summary>
		public float Radius 
		{
			get 
			{
				return m_radius;
			}
			set 
			{
				m_radius=value;
			}
		}

		/// <summary>
		/// Initializes a cone at the origin of height 2 and radius 1
		/// </summary>
		public Cone()
		{
			//
			// TODO: Add constructor logic here
			//
		}

		/// <summary>
		/// Initializes a cone at the origin
		/// </summary>
		/// <param name="Radius">Radius of the base of the cone</param>
		/// <param name="Height">Height of the cone</param>
		public Cone(float Radius, float Height) 
		{
			m_radius=Radius;
			m_height=Height;
		}

		/// <summary>
		/// Initializes a cone at the specified position, with the radius and height
		/// </summary>
		/// <param name="Position">Position of cone</param>
		/// <param name="Radius">Radius of base of cone</param>
		/// <param name="Height">Height of cone</param>
		public Cone(Vector3f Position, float Radius, float Height) 
		{
			m_position = Position;
			m_radius=Radius;
			m_height=Height;
		}

		/// <summary>
		/// Initializes a new cone with the given position, up vector, radius, and height
		/// </summary>
		/// <param name="Position">Position of cone (center of base)</param>
		/// <param name="Up">Vector pointing in cone's "up" direction</param>
		/// <param name="Radius">Radius of base of cone</param>
		/// <param name="Height">Height of cone</param>
		public Cone(Vector3f Position, Vector3f Up, float Radius, float Height) 
		{
			m_position=Position;
			m_baseNormal=Up;
			m_baseNormal.Normalize();
			m_radius=Radius;
			m_height=Height;
		}

		/// <summary>
		/// Calculates temporary variables for use in rendering
		/// </summary>
		public override void DoPreRenderCalculations()
		{

		}

		/// <summary>
		/// Gets the distance from the beginning of a ray to this cone.
		/// </summary>
		/// <param name="ray">The ray to trace</param>
		/// <returns>Distance from ray to cone, or negative if not intersection</returns>
		public override float DistanceToHit(BenWatson.BRayTracer.Raytracer.Ray ray)
		{
			float theta = 25.0f;
			float cosTheta2 = (float)Math.Cos(theta);
			cosTheta2*=cosTheta2;

			Vector3f A = (Vector3f)m_baseNormal.Clone();
			
			float M = A.X*A.X + A.Y * A.Y + A.Z*A.Z - cosTheta2;

			Vector3f D = ray.Direction;
			Vector3f delta = ray.Start - new Vector3f(0.0f,1.0f,0.0f);

			float c2 = (D.X * D.X  + D.Y * D.Y  + D.Z*D.Z) * M;
			float c1 = (D.X * delta.X + D.Y * delta.Y + D.Z * delta.Z)*M;
			float c0 = (delta.X * delta.X + delta.Y*delta.Y + delta.Z*delta.Z)*M;

			float t1 =0.0f, t2=0.0f;
			if (c2 !=0) 
			{
				float d = c1*c1-c0*c2;
				if (d>0.0f) 
				{
					d=(float)Math.Sqrt(d);
                    t1 = (-c1 + d)/c2;
					float z = ray.Start.Z + (ray.Direction.Z * t1);
					if (t1>0.0f && z>=0.0f)
						return t1;

					t2 = (-c1 - d)/c2;
					z = ray.Start.Z + ray.Direction.Z * t2;
					if (t2>0.0f && z>=0.0f)
						return t2;
				}
			}
			return -1.0f;
					

			/*float a = ray.Direction.X * ray.Direction.X + ray.Direction.Y * ray.Direction.Y - ray.Direction.Z*ray.Direction.Z;
			float b = ray.Direction.X * ray.Start.X + ray.Direction.Y*ray.Start.Y - ray.Direction.Z*ray.Start.Z;
			float c = ray.Start.X*ray.Start.X + ray.Start.Y*ray.Start.Y-ray.Start.Z*ray.Start.Z;
            
			float eps = 1.0e-10f;

			float tol = 1.0e-6f;

			if (Math.Abs(a) < eps) 
			{
				if (Math.Abs(b) >eps) 
				{
					float t1 = -0.5f *c/b;
					float z = ray.Start.Z+t1*ray.Direction.Z;
					if (t1>tol && z<=1.0f)
						return t1/ray.Direction.Magnitude;
				}
			} 
			else 
			{
				//check sides
				float d = b*b-a*c;
				if (d>=0.0f) 
				{
					d=(float)Math.Sqrt(d);

					float t1 = (-b-d)/a;
					float t2 = (-b+d)/a;

					float z=ray.Start.Z+t1*ray.Direction.Z;
					if (t1 > tol && z>=0.0f && z<=1.0f)
						return t1;

					z=ray.Start.Z+t2*ray.Direction.Z;
					if (z>=0.0f && z<=1.0f)
						return t2;
				}
			}

			return -1.0f;*/
		}

		/// <summary>
		/// Decides if ray hits cone or not.
		/// </summary>
		/// <param name="ray">Ray to trace</param>
		/// <returns>True if ray hits cone, false otherwise</returns>
		public override bool IsHit(BenWatson.BRayTracer.Raytracer.Ray ray)
		{
			return (DistanceToHit(ray) >=0.0f);
		}

		/// <summary>
		/// Gets the surface normal at the specified point.
		/// </summary>
		/// <param name="Point">Point at which to evaluate normal</param>
		/// <returns>Surface normal at point</returns>
		public override Vector3f NormalAtPoint(Vector3f Point)
		{
			return new Vector3f(0.0f,1.0f,0.0f);
		}



	}
}
