/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Implementation for Checkerboard primitive
 *
 * Revision History:
 *		12/20/2004 - Ben Watson - Added to VSS
 *		12/21/2004 - Ben Watson - Added XML comments
 *		12/29/2004 - Ben Watson - Added attributes for property editor.
 *		12/29/2004 - Ben Watson - Fixed tile size - clamps to >= 0.0
 *
 *--*/

using System;
using BenWatson.BRayTracer.Raytracer;
using BenWatson.BRayTracer;
using System.Drawing;
using System.ComponentModel;

namespace BenWatson.BRayTracer.Primitives
{
	/// <summary>
	/// A two-tone checkerboard pattern on a plane.
	/// </summary>
	/// <remarks></remarks>
	[Serializable]
	public class Checkerboard : Plane
	{
		/// <summary>
		/// Size of the tile
		/// </summary>
		protected float m_tileSize = 1.0f;
		/// <summary>
		/// Material for the secondary tiles
		/// </summary>
		protected Material m_tileMaterial = null;

		/// <summary>
		/// Default tile color
		/// </summary>
		protected static Color4f DefaultTileColor = Color.Green;

		#region Public Properties
		/// <summary>
		/// Gets or sets the material for the tile
		/// </summary>
		[Editable, Category("Checkerboard"), Description("Material 2 (The shape's primary material is Material 1)")]
		public Material TileMaterial 
		{
			get 
			{
				return m_tileMaterial;
			}
			set 
			{
				m_tileMaterial = value;
			}
		}

		/// <summary>
		/// Gets or sets the tile size
		/// </summary>
		[Viewable, Category("Checkerboard"), Description("Size of the tiles")]
		public float TileSize 
		{
			get 
			{
				return m_tileSize;
			}
			set 
			{
				m_tileSize = value;
				if (m_tileSize < 0.0f)
					m_tileSize=0.0f;
			}
		}

		#endregion

		/// <summary>
		/// Initializes a new checkerboard at the origin with the default tile color
		/// </summary>
		public Checkerboard() 
		{
			m_tileMaterial = new Material(DefaultTileColor);
		}

		/// <summary>
		/// Initializes a new checkerboard
		/// </summary>
		/// <param name="Position">Position of the plane</param>
		public Checkerboard(Vector3f Position) : base(Position,new Vector3f(0.0f,1.0f,0.0f))
		{
			m_tileMaterial = new Material(DefaultTileColor);
		}

		/// <summary>
		/// Initializes a new checkerboard
		/// </summary>
		/// <param name="Position">Position of the plane</param>
		/// <param name="TileSize">Tile size</param>
		public Checkerboard(Vector3f Position, float TileSize) :base(Position, new Vector3f(0.0f,1.0f,0.0f))
		{
			m_tileSize=TileSize;
			m_tileMaterial = new Material(DefaultTileColor);
		}

		/// <summary>
		/// Returns the material for the specific point
		/// </summary>
		/// <param name="Point">Point at which to evaluate material</param>
		/// <returns>A material</returns>
		public override Material GetMaterialAtPoint(Vector3f Point)
		{
			int i = (int)((Point.X - m_position.X) / m_tileSize) + 1;
			int j = (int)((Point.Z - m_position.Z) / m_tileSize) + 1;

			if (i<0) i*=-1;
			if (j<0) j*=-1;

			if (i%2 == j%2)
				return m_tileMaterial;

			return m_material;
		}

	}
}
