/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     MainForm controls all of the acvitity in the application. It contains all the panels, subwindows,
 * menus, toolbars, status bars, tabs, and other UI elements. It handles mouse clicks in the scene tree view,
 * as well as controlling the rendering procedure.
 *		There are a number of different sections of the code: 
 *
 * Revision History:
 *		12/20/2004 - Added to VSS
 *		12/22/2004 - Added XML comments
 *		12/26/2004 - Added render priority menus and handlers
 *
 *--*/

using System;
using System.Reflection;
using System.Threading;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;

using BenWatson.BRayTracer.Primitives;
using BenWatson.BRayTracer.Raytracer;
using BenWatson.BRayTracer.Editors;
using BenWatson.BRayTracer;

namespace BenWatson.BRayTracer.Forms
{
	/// <summary>
	/// Summary description for MainForm.
	/// </summary>
	public class MainForm : System.Windows.Forms.Form
	{

		#region UI elements

		
		private System.Windows.Forms.StatusBar statusBar1;
		private System.Windows.Forms.ToolBar toolBar1;
		private System.Windows.Forms.MainMenu mainMenu1;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.MenuItem menuItemFileExit;
		private System.Windows.Forms.TabControl tabControl1;
		private System.Windows.Forms.Splitter splitter1;
		private System.Windows.Forms.TabPage tabPageScene;
		private System.Windows.Forms.TabPage tabPageStats;
		private System.Windows.Forms.ToolBarButton toolBarButtonNewScene;
		private System.Windows.Forms.Panel panel1;
		private System.Windows.Forms.PictureBox pictureBox1;
		private System.Windows.Forms.TreeView treeViewScene;
		private System.Windows.Forms.ToolBarButton toolBarButtonSave;
		private System.ComponentModel.IContainer components;

		
		private System.Windows.Forms.ToolBarButton toolBarButtonStart;
		private System.Windows.Forms.ToolBarButton toolBarButtonStop;

		
		
		private System.Windows.Forms.StatusBarPanel statusBarPanelGeneral;
		private System.Windows.Forms.StatusBarPanel statusBarPanelElapsedTime;
		private System.Windows.Forms.StatusBarPanel statusBarPanelRaysPerSecond;
		private System.Windows.Forms.StatusBarPanel statusBarPanelPixelsPerSecond;
		private System.Windows.Forms.StatusBarPanel statusBarPanelRaysTraced;

		
		private System.Windows.Forms.ComboBox comboBoxImageSize;
		private System.Windows.Forms.Label label1;
		private System.Windows.Forms.ToolBarButton toolBarButtonAntiAlias;
		private System.Windows.Forms.ToolBarButton seperator1;
		private System.Windows.Forms.ToolBarButton toolBarButtonSaveOutput;
		private System.Windows.Forms.ToolBarButton seperator2;
		
		private System.Windows.Forms.ImageList imageListToolbar;
		private System.Windows.Forms.ToolBarButton toolBarButtonOpenScene;
		private System.Windows.Forms.Label label2;
		private System.Windows.Forms.ComboBox comboBoxDepth;
		private System.Windows.Forms.ContextMenu contextMenuShapeCollection;
		private System.Windows.Forms.MenuItem menuItemRemoveAllShapes;
		

		
		private System.Windows.Forms.ImageList imageListSceneTree;

		private System.Windows.Forms.ContextMenu contextMenuSceneObject;
		private System.Windows.Forms.MenuItem menuItemEditSceneObject;
		private System.Windows.Forms.MenuItem menuItem3;
		private System.Windows.Forms.MenuItem menuItemRemoveSceneObject;
		private System.Windows.Forms.ContextMenu contextMenuLights;
		private System.Windows.Forms.MenuItem menuItemAddLight;
		private System.Windows.Forms.MenuItem menuItem2;
		private System.Windows.Forms.MenuItem menuItemRemoveAllLights;
		private System.Windows.Forms.MenuItem menuItem4;
		private System.Windows.Forms.MenuItem menuItemNewScene;
		private System.Windows.Forms.MenuItem menuItemOpenScene;
		private System.Windows.Forms.MenuItem menuItem7;
		private System.Windows.Forms.MenuItem menuItemSaveScene;
		private System.Windows.Forms.MenuItem menuItemSaveSceneAs;

		
		private System.Windows.Forms.MenuItem mainMenuItemCamera;
		private System.Windows.Forms.MenuItem mainMenuItemLights;
		private System.Windows.Forms.MenuItem mainMenuItemShapes;
		private System.Windows.Forms.RichTextBox richTextBoxStats;
		private System.Windows.Forms.Label label3;
		private System.Windows.Forms.ComboBox comboBoxSubpixels;

		
		private TreeNode shapesNode, lightsNode, cameraNode;
		private MenuItem menuItemAddSphere, menuItemAddPolygon, menuItemAddPlane,menuItemAddCone, 
			menuItemAddPolygonMesh, menuItemAddCylinder, menuItemAddCheckerboard;


		#endregion

		#region Private Fields
		/// <summary>
		/// Associates the menu items with scene objects
		/// </summary>
		private Hashtable menuShapes= new Hashtable();
		/// <summary>
		/// Flag to indicate if the file needs saved
		/// </summary>
		private bool m_isDirty=false;

		/// <summary>
		/// Associates image formats with their filename filters
		/// </summary>
		private ImageFormatInfo[] imageFormats=null;
		/// <summary>
		/// Associates scene object types with their editor information
		/// </summary>
		private Hashtable typeEditors=new Hashtable();
		/// <summary>
		/// Contains settings for the ray tracer
		/// </summary>
		private RayTraceSettings traceSettings = new RayTraceSettings();
		/// <summary>
		/// Contains program settings
		/// </summary>
		private ProgramSettings programSettings = new ProgramSettings();

		/// <summary>
		/// Describes image size presets available
		/// </summary>
		private ImageSize[] imageSizes=null;
		/// <summary>
		/// The current working scene
		/// </summary>
		/// <remarks>Only one scene may be open at a time.</remarks>
		private Scene m_scene = null;
		/// <summary>
		/// The output bitmap
		/// </summary>
		private Bitmap bmp = null;
		/// <summary>
		/// Lock object for access to the output bitmap
		/// </summary>
		private object bmpLock = new object();
		/// <summary>
		/// Lock object for access to the scene
		/// </summary>
		private object sceneLock = new object();
		/// <summary>
		/// The ray tracer
		/// </summary>
		private Tracer tracer = null;
		/// <summary>
		/// Flag to indicate whether the ray tracer is currently rendering the scene
		/// </summary>
		private bool bRendering = false;
		/// <summary>
		/// Signal for the end of the rendering
		/// </summary>
		private ManualResetEvent renderProcDone = new ManualResetEvent(false);
		private System.Windows.Forms.MenuItem menuItem5;
		private System.Windows.Forms.MenuItem menuItemAbout;
		private System.Windows.Forms.MenuItem menuItem6;
		private System.Windows.Forms.MenuItem menuItemRenderStart;
		private System.Windows.Forms.MenuItem menuItem8;
		private System.Windows.Forms.MenuItem menuItemRenderPreferences;
		private System.Windows.Forms.MenuItem menuItem9;
		private System.Windows.Forms.MenuItem menuItem10;
		private System.Windows.Forms.MenuItem menuItemRenderPriorityHigh;
		private System.Windows.Forms.MenuItem menuItemRenderPriorityAboveNormal;
		private System.Windows.Forms.MenuItem menuItemRenderPriorityNormal;
		private System.Windows.Forms.MenuItem menuItemRenderPriorityBelowNormal;
		private System.Windows.Forms.MenuItem menuItemRenderPriorityLow;
		private System.Windows.Forms.MenuItem menuItemRenderStop;
		/// <summary>
		/// The render thread
		/// </summary>
		private Thread renderThread=null;


		#endregion

		#region Events
		/// <summary>
		/// Notifies listeners when rendering has started
		/// </summary>
		public event EventHandler RenderStarted;
		/// <summary>
		/// Notifies listeners when rendering has stopped or been canceled
		/// </summary>
		public event EventHandler RenderFinished;
		#endregion


		/// <summary>
		/// Constructs the main form and calls initalizing functions
		/// </summary>
		/// <remarks>The constructor first the GUI, then builds the shape menus. 
		/// It builds image exporters, registers shape editors, and loads the default scene and settings.</remarks>
		public MainForm()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			BuildMenus();

			BuildImageExporters();

			RegisterEditors();

			BuildImageSizes();

			LoadDefaultScene();

			LoadDefaultSettings();
		
			this.RenderStarted+=new EventHandler(MainForm_RenderStarted);
			this.RenderFinished+=new EventHandler(MainForm_RenderFinished);

		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(MainForm));
			this.statusBar1 = new System.Windows.Forms.StatusBar();
			this.statusBarPanelGeneral = new System.Windows.Forms.StatusBarPanel();
			this.statusBarPanelElapsedTime = new System.Windows.Forms.StatusBarPanel();
			this.statusBarPanelRaysTraced = new System.Windows.Forms.StatusBarPanel();
			this.statusBarPanelRaysPerSecond = new System.Windows.Forms.StatusBarPanel();
			this.statusBarPanelPixelsPerSecond = new System.Windows.Forms.StatusBarPanel();
			this.toolBar1 = new System.Windows.Forms.ToolBar();
			this.toolBarButtonNewScene = new System.Windows.Forms.ToolBarButton();
			this.toolBarButtonOpenScene = new System.Windows.Forms.ToolBarButton();
			this.toolBarButtonSave = new System.Windows.Forms.ToolBarButton();
			this.toolBarButtonSaveOutput = new System.Windows.Forms.ToolBarButton();
			this.seperator1 = new System.Windows.Forms.ToolBarButton();
			this.toolBarButtonStart = new System.Windows.Forms.ToolBarButton();
			this.toolBarButtonStop = new System.Windows.Forms.ToolBarButton();
			this.seperator2 = new System.Windows.Forms.ToolBarButton();
			this.toolBarButtonAntiAlias = new System.Windows.Forms.ToolBarButton();
			this.imageListToolbar = new System.Windows.Forms.ImageList(this.components);
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.menuItemNewScene = new System.Windows.Forms.MenuItem();
			this.menuItemOpenScene = new System.Windows.Forms.MenuItem();
			this.menuItemSaveScene = new System.Windows.Forms.MenuItem();
			this.menuItemSaveSceneAs = new System.Windows.Forms.MenuItem();
			this.menuItem7 = new System.Windows.Forms.MenuItem();
			this.menuItemFileExit = new System.Windows.Forms.MenuItem();
			this.mainMenuItemCamera = new System.Windows.Forms.MenuItem();
			this.mainMenuItemLights = new System.Windows.Forms.MenuItem();
			this.mainMenuItemShapes = new System.Windows.Forms.MenuItem();
			this.menuItem6 = new System.Windows.Forms.MenuItem();
			this.menuItemRenderStart = new System.Windows.Forms.MenuItem();
			this.menuItemRenderStop = new System.Windows.Forms.MenuItem();
			this.menuItem8 = new System.Windows.Forms.MenuItem();
			this.menuItemRenderPreferences = new System.Windows.Forms.MenuItem();
			this.menuItem9 = new System.Windows.Forms.MenuItem();
			this.menuItem10 = new System.Windows.Forms.MenuItem();
			this.menuItemRenderPriorityHigh = new System.Windows.Forms.MenuItem();
			this.menuItemRenderPriorityAboveNormal = new System.Windows.Forms.MenuItem();
			this.menuItemRenderPriorityNormal = new System.Windows.Forms.MenuItem();
			this.menuItemRenderPriorityBelowNormal = new System.Windows.Forms.MenuItem();
			this.menuItemRenderPriorityLow = new System.Windows.Forms.MenuItem();
			this.menuItem5 = new System.Windows.Forms.MenuItem();
			this.menuItemAbout = new System.Windows.Forms.MenuItem();
			this.tabControl1 = new System.Windows.Forms.TabControl();
			this.tabPageScene = new System.Windows.Forms.TabPage();
			this.treeViewScene = new System.Windows.Forms.TreeView();
			this.imageListSceneTree = new System.Windows.Forms.ImageList(this.components);
			this.tabPageStats = new System.Windows.Forms.TabPage();
			this.richTextBoxStats = new System.Windows.Forms.RichTextBox();
			this.splitter1 = new System.Windows.Forms.Splitter();
			this.panel1 = new System.Windows.Forms.Panel();
			this.pictureBox1 = new System.Windows.Forms.PictureBox();
			this.comboBoxImageSize = new System.Windows.Forms.ComboBox();
			this.label1 = new System.Windows.Forms.Label();
			this.label2 = new System.Windows.Forms.Label();
			this.comboBoxDepth = new System.Windows.Forms.ComboBox();
			this.contextMenuShapeCollection = new System.Windows.Forms.ContextMenu();
			this.menuItem4 = new System.Windows.Forms.MenuItem();
			this.menuItemRemoveAllShapes = new System.Windows.Forms.MenuItem();
			this.contextMenuSceneObject = new System.Windows.Forms.ContextMenu();
			this.menuItemEditSceneObject = new System.Windows.Forms.MenuItem();
			this.menuItem3 = new System.Windows.Forms.MenuItem();
			this.menuItemRemoveSceneObject = new System.Windows.Forms.MenuItem();
			this.contextMenuLights = new System.Windows.Forms.ContextMenu();
			this.menuItemAddLight = new System.Windows.Forms.MenuItem();
			this.menuItem2 = new System.Windows.Forms.MenuItem();
			this.menuItemRemoveAllLights = new System.Windows.Forms.MenuItem();
			this.label3 = new System.Windows.Forms.Label();
			this.comboBoxSubpixels = new System.Windows.Forms.ComboBox();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelGeneral)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelElapsedTime)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelRaysTraced)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelRaysPerSecond)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelPixelsPerSecond)).BeginInit();
			this.tabControl1.SuspendLayout();
			this.tabPageScene.SuspendLayout();
			this.tabPageStats.SuspendLayout();
			this.panel1.SuspendLayout();
			this.SuspendLayout();
			// 
			// statusBar1
			// 
			this.statusBar1.Location = new System.Drawing.Point(0, 419);
			this.statusBar1.Name = "statusBar1";
			this.statusBar1.Panels.AddRange(new System.Windows.Forms.StatusBarPanel[] {
																						  this.statusBarPanelGeneral,
																						  this.statusBarPanelElapsedTime,
																						  this.statusBarPanelRaysTraced,
																						  this.statusBarPanelRaysPerSecond,
																						  this.statusBarPanelPixelsPerSecond});
			this.statusBar1.ShowPanels = true;
			this.statusBar1.Size = new System.Drawing.Size(720, 22);
			this.statusBar1.TabIndex = 8;
			this.statusBar1.Text = "statusBar1";
			// 
			// statusBarPanelGeneral
			// 
			this.statusBarPanelGeneral.Text = "Ready";
			this.statusBarPanelGeneral.ToolTipText = "Status";
			// 
			// statusBarPanelElapsedTime
			// 
			this.statusBarPanelElapsedTime.ToolTipText = "Elapased Time";
			this.statusBarPanelElapsedTime.Width = 130;
			// 
			// statusBarPanelRaysTraced
			// 
			this.statusBarPanelRaysTraced.ToolTipText = "Number of rays traced";
			this.statusBarPanelRaysTraced.Width = 130;
			// 
			// statusBarPanelRaysPerSecond
			// 
			this.statusBarPanelRaysPerSecond.ToolTipText = "Rays per second";
			this.statusBarPanelRaysPerSecond.Width = 130;
			// 
			// statusBarPanelPixelsPerSecond
			// 
			this.statusBarPanelPixelsPerSecond.ToolTipText = "Pixels per second";
			this.statusBarPanelPixelsPerSecond.Width = 130;
			// 
			// toolBar1
			// 
			this.toolBar1.Buttons.AddRange(new System.Windows.Forms.ToolBarButton[] {
																						this.toolBarButtonNewScene,
																						this.toolBarButtonOpenScene,
																						this.toolBarButtonSave,
																						this.toolBarButtonSaveOutput,
																						this.seperator1,
																						this.toolBarButtonStart,
																						this.toolBarButtonStop,
																						this.seperator2,
																						this.toolBarButtonAntiAlias});
			this.toolBar1.DropDownArrows = true;
			this.toolBar1.ImageList = this.imageListToolbar;
			this.toolBar1.Location = new System.Drawing.Point(0, 0);
			this.toolBar1.Name = "toolBar1";
			this.toolBar1.ShowToolTips = true;
			this.toolBar1.Size = new System.Drawing.Size(720, 44);
			this.toolBar1.TabIndex = 0;
			this.toolBar1.ButtonClick += new System.Windows.Forms.ToolBarButtonClickEventHandler(this.toolBar1_ButtonClick);
			// 
			// toolBarButtonNewScene
			// 
			this.toolBarButtonNewScene.ImageIndex = 0;
			this.toolBarButtonNewScene.ToolTipText = "Create a New Scene";
			// 
			// toolBarButtonOpenScene
			// 
			this.toolBarButtonOpenScene.ImageIndex = 1;
			this.toolBarButtonOpenScene.ToolTipText = "Open a scene from an existing file";
			// 
			// toolBarButtonSave
			// 
			this.toolBarButtonSave.ImageIndex = 2;
			this.toolBarButtonSave.ToolTipText = "Save this scene to a file";
			// 
			// toolBarButtonSaveOutput
			// 
			this.toolBarButtonSaveOutput.ImageIndex = 3;
			this.toolBarButtonSaveOutput.ToolTipText = "Save output to an image file";
			// 
			// seperator1
			// 
			this.seperator1.Style = System.Windows.Forms.ToolBarButtonStyle.Separator;
			// 
			// toolBarButtonStart
			// 
			this.toolBarButtonStart.ImageIndex = 4;
			this.toolBarButtonStart.ToolTipText = "Start Rendering";
			// 
			// toolBarButtonStop
			// 
			this.toolBarButtonStop.Enabled = false;
			this.toolBarButtonStop.ImageIndex = 5;
			this.toolBarButtonStop.ToolTipText = "Stop rendering";
			// 
			// seperator2
			// 
			this.seperator2.Style = System.Windows.Forms.ToolBarButtonStyle.Separator;
			// 
			// toolBarButtonAntiAlias
			// 
			this.toolBarButtonAntiAlias.ImageIndex = 6;
			this.toolBarButtonAntiAlias.Style = System.Windows.Forms.ToolBarButtonStyle.ToggleButton;
			this.toolBarButtonAntiAlias.ToolTipText = "Toggle Antialiasing";
			// 
			// imageListToolbar
			// 
			this.imageListToolbar.ImageSize = new System.Drawing.Size(32, 32);
			this.imageListToolbar.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imageListToolbar.ImageStream")));
			this.imageListToolbar.TransparentColor = System.Drawing.Color.White;
			// 
			// mainMenu1
			// 
			this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuItem1,
																					  this.mainMenuItemCamera,
																					  this.mainMenuItemLights,
																					  this.mainMenuItemShapes,
																					  this.menuItem6,
																					  this.menuItem5});
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 0;
			this.menuItem1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuItemNewScene,
																					  this.menuItemOpenScene,
																					  this.menuItemSaveScene,
																					  this.menuItemSaveSceneAs,
																					  this.menuItem7,
																					  this.menuItemFileExit});
			this.menuItem1.Text = "&File";
			// 
			// menuItemNewScene
			// 
			this.menuItemNewScene.Index = 0;
			this.menuItemNewScene.Shortcut = System.Windows.Forms.Shortcut.CtrlN;
			this.menuItemNewScene.Text = "&New Scene";
			// 
			// menuItemOpenScene
			// 
			this.menuItemOpenScene.Index = 1;
			this.menuItemOpenScene.Shortcut = System.Windows.Forms.Shortcut.CtrlO;
			this.menuItemOpenScene.Text = "&Open Scene...";
			// 
			// menuItemSaveScene
			// 
			this.menuItemSaveScene.Index = 2;
			this.menuItemSaveScene.Shortcut = System.Windows.Forms.Shortcut.CtrlS;
			this.menuItemSaveScene.Text = "&Save Scene";
			// 
			// menuItemSaveSceneAs
			// 
			this.menuItemSaveSceneAs.Index = 3;
			this.menuItemSaveSceneAs.Shortcut = System.Windows.Forms.Shortcut.CtrlK;
			this.menuItemSaveSceneAs.Text = "S&ave Scene as...";
			// 
			// menuItem7
			// 
			this.menuItem7.Index = 4;
			this.menuItem7.Text = "-";
			// 
			// menuItemFileExit
			// 
			this.menuItemFileExit.Index = 5;
			this.menuItemFileExit.Shortcut = System.Windows.Forms.Shortcut.AltF4;
			this.menuItemFileExit.Text = "E&xit";
			this.menuItemFileExit.Click += new System.EventHandler(this.menuItemFileExit_Click);
			// 
			// mainMenuItemCamera
			// 
			this.mainMenuItemCamera.Index = 1;
			this.mainMenuItemCamera.Text = "&Camera";
			// 
			// mainMenuItemLights
			// 
			this.mainMenuItemLights.Index = 2;
			this.mainMenuItemLights.Text = "&Lights";
			// 
			// mainMenuItemShapes
			// 
			this.mainMenuItemShapes.Index = 3;
			this.mainMenuItemShapes.Text = "&Shapes";
			// 
			// menuItem6
			// 
			this.menuItem6.Index = 4;
			this.menuItem6.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuItemRenderStart,
																					  this.menuItemRenderStop,
																					  this.menuItem8,
																					  this.menuItemRenderPreferences,
																					  this.menuItem9,
																					  this.menuItem10});
			this.menuItem6.Text = "&Render";
			// 
			// menuItemRenderStart
			// 
			this.menuItemRenderStart.Index = 0;
			this.menuItemRenderStart.Text = "&Start";
			this.menuItemRenderStart.Click += new System.EventHandler(this.menuItemRenderStart_Click);
			// 
			// menuItemRenderStop
			// 
			this.menuItemRenderStop.Enabled = false;
			this.menuItemRenderStop.Index = 1;
			this.menuItemRenderStop.Text = "S&top";
			this.menuItemRenderStop.Click += new System.EventHandler(this.menuItemRenderStop_Click);
			// 
			// menuItem8
			// 
			this.menuItem8.Index = 2;
			this.menuItem8.Text = "-";
			// 
			// menuItemRenderPreferences
			// 
			this.menuItemRenderPreferences.Enabled = false;
			this.menuItemRenderPreferences.Index = 3;
			this.menuItemRenderPreferences.Text = "Render &Preferences...";
			// 
			// menuItem9
			// 
			this.menuItem9.Index = 4;
			this.menuItem9.Text = "-";
			// 
			// menuItem10
			// 
			this.menuItem10.Index = 5;
			this.menuItem10.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					   this.menuItemRenderPriorityHigh,
																					   this.menuItemRenderPriorityAboveNormal,
																					   this.menuItemRenderPriorityNormal,
																					   this.menuItemRenderPriorityBelowNormal,
																					   this.menuItemRenderPriorityLow});
			this.menuItem10.Text = "&Render Priority";
			// 
			// menuItemRenderPriorityHigh
			// 
			this.menuItemRenderPriorityHigh.Index = 0;
			this.menuItemRenderPriorityHigh.RadioCheck = true;
			this.menuItemRenderPriorityHigh.Text = "&High";
			this.menuItemRenderPriorityHigh.Click += new System.EventHandler(this.OnRenderPriorityChanged);
			// 
			// menuItemRenderPriorityAboveNormal
			// 
			this.menuItemRenderPriorityAboveNormal.Index = 1;
			this.menuItemRenderPriorityAboveNormal.RadioCheck = true;
			this.menuItemRenderPriorityAboveNormal.Text = "&Above Normal";
			this.menuItemRenderPriorityAboveNormal.Click += new System.EventHandler(this.OnRenderPriorityChanged);
			// 
			// menuItemRenderPriorityNormal
			// 
			this.menuItemRenderPriorityNormal.Checked = true;
			this.menuItemRenderPriorityNormal.Index = 2;
			this.menuItemRenderPriorityNormal.RadioCheck = true;
			this.menuItemRenderPriorityNormal.Text = "&Normal";
			this.menuItemRenderPriorityNormal.Click += new System.EventHandler(this.OnRenderPriorityChanged);
			// 
			// menuItemRenderPriorityBelowNormal
			// 
			this.menuItemRenderPriorityBelowNormal.Index = 3;
			this.menuItemRenderPriorityBelowNormal.RadioCheck = true;
			this.menuItemRenderPriorityBelowNormal.Text = "&Below Normal";
			this.menuItemRenderPriorityBelowNormal.Click += new System.EventHandler(this.OnRenderPriorityChanged);
			// 
			// menuItemRenderPriorityLow
			// 
			this.menuItemRenderPriorityLow.Index = 4;
			this.menuItemRenderPriorityLow.RadioCheck = true;
			this.menuItemRenderPriorityLow.Text = "&Low";
			this.menuItemRenderPriorityLow.Click += new System.EventHandler(this.OnRenderPriorityChanged);
			// 
			// menuItem5
			// 
			this.menuItem5.Index = 5;
			this.menuItem5.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuItemAbout});
			this.menuItem5.Text = "&Help";
			// 
			// menuItemAbout
			// 
			this.menuItemAbout.Index = 0;
			this.menuItemAbout.Text = "&About...";
			this.menuItemAbout.Click += new System.EventHandler(this.menuItemAbout_Click);
			// 
			// tabControl1
			// 
			this.tabControl1.Controls.Add(this.tabPageScene);
			this.tabControl1.Controls.Add(this.tabPageStats);
			this.tabControl1.Dock = System.Windows.Forms.DockStyle.Left;
			this.tabControl1.Location = new System.Drawing.Point(0, 44);
			this.tabControl1.Name = "tabControl1";
			this.tabControl1.SelectedIndex = 0;
			this.tabControl1.Size = new System.Drawing.Size(216, 375);
			this.tabControl1.TabIndex = 6;
			// 
			// tabPageScene
			// 
			this.tabPageScene.Controls.Add(this.treeViewScene);
			this.tabPageScene.Location = new System.Drawing.Point(4, 22);
			this.tabPageScene.Name = "tabPageScene";
			this.tabPageScene.Size = new System.Drawing.Size(208, 349);
			this.tabPageScene.TabIndex = 0;
			this.tabPageScene.Text = "Scene";
			// 
			// treeViewScene
			// 
			this.treeViewScene.Dock = System.Windows.Forms.DockStyle.Fill;
			this.treeViewScene.ImageList = this.imageListSceneTree;
			this.treeViewScene.Location = new System.Drawing.Point(0, 0);
			this.treeViewScene.Name = "treeViewScene";
			this.treeViewScene.Size = new System.Drawing.Size(208, 349);
			this.treeViewScene.TabIndex = 0;
			this.treeViewScene.MouseUp += new System.Windows.Forms.MouseEventHandler(this.OnSceneTree_MouseUp);
			this.treeViewScene.DoubleClick += new System.EventHandler(this.OnSceneTreeDoubleClick);
			// 
			// imageListSceneTree
			// 
			this.imageListSceneTree.ColorDepth = System.Windows.Forms.ColorDepth.Depth16Bit;
			this.imageListSceneTree.ImageSize = new System.Drawing.Size(16, 16);
			this.imageListSceneTree.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imageListSceneTree.ImageStream")));
			this.imageListSceneTree.TransparentColor = System.Drawing.Color.White;
			// 
			// tabPageStats
			// 
			this.tabPageStats.Controls.Add(this.richTextBoxStats);
			this.tabPageStats.Location = new System.Drawing.Point(4, 22);
			this.tabPageStats.Name = "tabPageStats";
			this.tabPageStats.Size = new System.Drawing.Size(208, 349);
			this.tabPageStats.TabIndex = 1;
			this.tabPageStats.Text = "Statistics";
			// 
			// richTextBoxStats
			// 
			this.richTextBoxStats.AcceptsTab = true;
			this.richTextBoxStats.AutoSize = true;
			this.richTextBoxStats.Dock = System.Windows.Forms.DockStyle.Fill;
			this.richTextBoxStats.Location = new System.Drawing.Point(0, 0);
			this.richTextBoxStats.Name = "richTextBoxStats";
			this.richTextBoxStats.Size = new System.Drawing.Size(208, 349);
			this.richTextBoxStats.TabIndex = 0;
			this.richTextBoxStats.Text = "Sample Text";
			// 
			// splitter1
			// 
			this.splitter1.Location = new System.Drawing.Point(216, 44);
			this.splitter1.Name = "splitter1";
			this.splitter1.Size = new System.Drawing.Size(3, 375);
			this.splitter1.TabIndex = 3;
			this.splitter1.TabStop = false;
			// 
			// panel1
			// 
			this.panel1.Controls.Add(this.pictureBox1);
			this.panel1.Dock = System.Windows.Forms.DockStyle.Fill;
			this.panel1.Location = new System.Drawing.Point(219, 44);
			this.panel1.Name = "panel1";
			this.panel1.Size = new System.Drawing.Size(501, 375);
			this.panel1.TabIndex = 7;
			// 
			// pictureBox1
			// 
			this.pictureBox1.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.pictureBox1.Dock = System.Windows.Forms.DockStyle.Fill;
			this.pictureBox1.Location = new System.Drawing.Point(0, 0);
			this.pictureBox1.Name = "pictureBox1";
			this.pictureBox1.Size = new System.Drawing.Size(501, 375);
			this.pictureBox1.TabIndex = 0;
			this.pictureBox1.TabStop = false;
			this.pictureBox1.Paint += new System.Windows.Forms.PaintEventHandler(this.OnPicturePaint);
			// 
			// comboBoxImageSize
			// 
			this.comboBoxImageSize.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboBoxImageSize.Location = new System.Drawing.Point(608, 16);
			this.comboBoxImageSize.Name = "comboBoxImageSize";
			this.comboBoxImageSize.Size = new System.Drawing.Size(104, 21);
			this.comboBoxImageSize.TabIndex = 5;
			this.comboBoxImageSize.SelectionChangeCommitted += new System.EventHandler(this.OnImageSizeSelectionChanged);
			// 
			// label1
			// 
			this.label1.Location = new System.Drawing.Point(544, 18);
			this.label1.Name = "label1";
			this.label1.Size = new System.Drawing.Size(64, 16);
			this.label1.TabIndex = 4;
			this.label1.Text = "Image size:";
			// 
			// label2
			// 
			this.label2.Location = new System.Drawing.Point(416, 18);
			this.label2.Name = "label2";
			this.label2.Size = new System.Drawing.Size(72, 16);
			this.label2.TabIndex = 2;
			this.label2.Text = "Trace Depth:";
			// 
			// comboBoxDepth
			// 
			this.comboBoxDepth.Items.AddRange(new object[] {
															   "1",
															   "2",
															   "4",
															   "8",
															   "10",
															   "20",
															   "50",
															   "100",
															   "500",
															   "1000"});
			this.comboBoxDepth.Location = new System.Drawing.Point(488, 16);
			this.comboBoxDepth.Name = "comboBoxDepth";
			this.comboBoxDepth.Size = new System.Drawing.Size(48, 21);
			this.comboBoxDepth.TabIndex = 3;
			this.comboBoxDepth.Text = "10";
			this.comboBoxDepth.SelectedIndexChanged += new System.EventHandler(this.OnDepthChanged);
			this.comboBoxDepth.Leave += new System.EventHandler(this.OnDepthChanged);
			this.comboBoxDepth.SelectionChangeCommitted += new System.EventHandler(this.OnDepthChanged);
			// 
			// contextMenuShapeCollection
			// 
			this.contextMenuShapeCollection.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																									   this.menuItem4,
																									   this.menuItemRemoveAllShapes});
			// 
			// menuItem4
			// 
			this.menuItem4.Index = 0;
			this.menuItem4.Text = "-";
			// 
			// menuItemRemoveAllShapes
			// 
			this.menuItemRemoveAllShapes.Index = 1;
			this.menuItemRemoveAllShapes.Text = "Remove all Shapes";
			// 
			// contextMenuSceneObject
			// 
			this.contextMenuSceneObject.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																								   this.menuItemEditSceneObject,
																								   this.menuItem3,
																								   this.menuItemRemoveSceneObject});
			// 
			// menuItemEditSceneObject
			// 
			this.menuItemEditSceneObject.Index = 0;
			this.menuItemEditSceneObject.Text = "&Edit...";
			// 
			// menuItem3
			// 
			this.menuItem3.Index = 1;
			this.menuItem3.Text = "-";
			// 
			// menuItemRemoveSceneObject
			// 
			this.menuItemRemoveSceneObject.Index = 2;
			this.menuItemRemoveSceneObject.Text = "&Remove";
			// 
			// contextMenuLights
			// 
			this.contextMenuLights.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																							  this.menuItemAddLight,
																							  this.menuItem2,
																							  this.menuItemRemoveAllLights});
			// 
			// menuItemAddLight
			// 
			this.menuItemAddLight.Index = 0;
			this.menuItemAddLight.Text = "&Add Light...";
			this.menuItemAddLight.Click += new System.EventHandler(this.OnAddLight);
			// 
			// menuItem2
			// 
			this.menuItem2.Index = 1;
			this.menuItem2.Text = "-";
			// 
			// menuItemRemoveAllLights
			// 
			this.menuItemRemoveAllLights.Index = 2;
			this.menuItemRemoveAllLights.Text = "&Remove All Lights";
			// 
			// label3
			// 
			this.label3.Location = new System.Drawing.Point(296, 18);
			this.label3.Name = "label3";
			this.label3.Size = new System.Drawing.Size(64, 16);
			this.label3.TabIndex = 9;
			this.label3.Text = "Subpixels:";
			// 
			// comboBoxSubpixels
			// 
			this.comboBoxSubpixels.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboBoxSubpixels.Enabled = false;
			this.comboBoxSubpixels.Items.AddRange(new object[] {
																   "4",
																   "9",
																   "16",
																   "64",
																   "256",
																   "1024"});
			this.comboBoxSubpixels.Location = new System.Drawing.Point(352, 16);
			this.comboBoxSubpixels.Name = "comboBoxSubpixels";
			this.comboBoxSubpixels.Size = new System.Drawing.Size(56, 21);
			this.comboBoxSubpixels.TabIndex = 10;
			this.comboBoxSubpixels.SelectedIndexChanged += new System.EventHandler(this.OnSubpixelsChanged);
			this.comboBoxSubpixels.Leave += new System.EventHandler(this.OnSubpixelsChanged);
			this.comboBoxSubpixels.SelectionChangeCommitted += new System.EventHandler(this.OnSubpixelsChanged);
			// 
			// MainForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(720, 441);
			this.Controls.Add(this.comboBoxSubpixels);
			this.Controls.Add(this.label3);
			this.Controls.Add(this.comboBoxDepth);
			this.Controls.Add(this.label2);
			this.Controls.Add(this.label1);
			this.Controls.Add(this.comboBoxImageSize);
			this.Controls.Add(this.panel1);
			this.Controls.Add(this.splitter1);
			this.Controls.Add(this.tabControl1);
			this.Controls.Add(this.toolBar1);
			this.Controls.Add(this.statusBar1);
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.Menu = this.mainMenu1;
			this.Name = "MainForm";
			this.Text = "BRayTracer";
			this.Closing += new System.ComponentModel.CancelEventHandler(this.OnMainFormClosing);
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelGeneral)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelElapsedTime)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelRaysTraced)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelRaysPerSecond)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanelPixelsPerSecond)).EndInit();
			this.tabControl1.ResumeLayout(false);
			this.tabPageScene.ResumeLayout(false);
			this.tabPageStats.ResumeLayout(false);
			this.panel1.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		/// <remarks>Enables visual styles to give it an XP look</remarks>
		[STAThread]
		static void Main() 
		{
			Application.EnableVisualStyles();
			Application.DoEvents();
			Application.Run(new MainForm());
		}

		#region Initialization, building editors, menus, etc.
		/// <summary>
		/// Loads default settings for the tracer and the GUI
		/// </summary>
		private void LoadDefaultSettings() 
		{
			traceSettings.ImageWidth = 640;
			traceSettings.ImageHeight = 480;
			traceSettings.SubPixels = 4;
			traceSettings.AntiAlias=false;
			traceSettings.TraceLevels=10;
			traceSettings.MaxDepth = 1e6f;
			
			this.comboBoxSubpixels.SelectedIndex = 0;
			this.comboBoxDepth.Text =traceSettings.TraceLevels.ToString();

			programSettings.SceneFilenameFilter = "BRayTracer files (*.bray)|*.bray|All files (*.*)|*.*";
			programSettings.DefaultNodeColor = Color.DarkGray;
		}

		/// <summary>
		/// Builds image sizes and sets up the combo to display them
		/// </summary>
		private void BuildImageSizes() 
		{
			int numSizes = 6;
			imageSizes = new ImageSize[numSizes];
			
			imageSizes[0] = new ImageSize(320,240);
			imageSizes[1] = new ImageSize(640, 480);
			imageSizes[2] = new ImageSize(800,600);
			imageSizes[3] = new ImageSize(1024,768);
			imageSizes[4] = new ImageSize(1280,1024);
			imageSizes[5] = new ImageSize(1600,1200);

			//fill in box
			for (int i=0;i<numSizes;i++) 
			{
				this.comboBoxImageSize.Items.Add(imageSizes[i].ToString());
				this.comboBoxImageSize.Tag = imageSizes;
			}

			this.bmp = new Bitmap(640,480);

			this.comboBoxImageSize.SelectedIndex = 1;
		}

		/// <summary>
		/// Builds image exporters for use in exporting output
		/// </summary>
		private void BuildImageExporters() 
		{
            imageFormats = new ImageFormatInfo[4];
			imageFormats[0] = new ImageFormatInfo("Windows Bitmap (*.bmp)|*.bmp",System.Drawing.Imaging.ImageFormat.Bmp);
			imageFormats[1] = new ImageFormatInfo("JPEG (*.jpg)|*.jpg",System.Drawing.Imaging.ImageFormat.Jpeg);
			imageFormats[2] = new ImageFormatInfo("TIFF (*.tiff)|*.tiff",System.Drawing.Imaging.ImageFormat.Tiff);
			imageFormats[3] = new ImageFormatInfo("PNG (*.png)|*.png",System.Drawing.Imaging.ImageFormat.Png);

			System.Text.StringBuilder sb=  new System.Text.StringBuilder(100);
			for (int i=0;i<imageFormats.Length;i++) 
			{
                sb.Append(imageFormats[i].Filter);
				if (i<imageFormats.Length-1)
					sb.Append("|");
			}
			programSettings.ImageFilters = sb.ToString();
			
		}

		/// <summary>
		/// Builds shapes menus dynamically for the context menu and the main menu
		/// </summary>
		private void BuildMenus() 
		{
			menuItemAddSphere = new MenuItem("Add Sphere...",new EventHandler(OnAddNewShape));
			menuItemAddPolygon = new MenuItem("Add Polygon...",new EventHandler(OnAddNewShape));
			menuItemAddPolygonMesh = new MenuItem("Add Polygon Mesh",new EventHandler(OnAddNewShape));
			menuItemAddCone=new MenuItem("Add Cone...",new EventHandler(OnAddNewShape));
			menuItemAddPlane = new MenuItem("Add Plane...",new EventHandler(OnAddNewShape));
			menuItemAddCheckerboard = new MenuItem("Add Checkerboard...",new EventHandler(OnAddNewShape));
			menuItemAddCylinder = new MenuItem("Add Cylinder...",new EventHandler(OnAddNewShape));

			menuItemEditSceneObject.Click+=new EventHandler(OnEditSceneObject);
			menuItemRemoveSceneObject.Click+=new EventHandler(OnRemoveSceneObject);
			
			//add to context menu
			this.contextMenuShapeCollection.MenuItems.Add(0,menuItemAddSphere);
			this.contextMenuShapeCollection.MenuItems.Add(1,menuItemAddPlane);
			this.contextMenuShapeCollection.MenuItems.Add(2,menuItemAddCheckerboard);
			this.contextMenuShapeCollection.MenuItems.Add(2,menuItemAddPolygon);
			this.contextMenuShapeCollection.MenuItems.Add(3,menuItemAddPolygonMesh);
			this.contextMenuShapeCollection.MenuItems.Add(4,menuItemAddCone);
			this.contextMenuShapeCollection.MenuItems.Add(5,menuItemAddCylinder);

			//add to main menu
			/*this.mainMenuItemShapes.MenuItems.Add(0,menuItemAddSphere);
			this.mainMenuItemShapes.MenuItems.Add(1,menuItemAddPlane);
			this.mainMenuItemShapes.MenuItems.Add(2,menuItemAddPolygon);
			this.mainMenuItemShapes.MenuItems.Add(3,menuItemAddPolygonMesh);
			this.mainMenuItemShapes.MenuItems.Add(4,menuItemAddCone);*/
            
			
			//build menu - Shape relation
			this.menuShapes[menuItemAddSphere] = typeof(Sphere);
			this.menuShapes[menuItemAddPolygon] = typeof(Polygon);
			this.menuShapes[menuItemAddPolygonMesh] = typeof(PolygonMesh);
			this.menuShapes[menuItemAddPlane] = typeof(Plane);
			this.menuShapes[menuItemAddCone] = typeof(Cone);
			this.menuShapes[menuItemAddCylinder] = typeof(Cylinder);
			this.menuShapes[menuItemAddCheckerboard] = typeof(Checkerboard);
		}


		/// <summary>
		/// Associates editors for each scene object type
		/// </summary>
		/// <remarks>A scene object that does not have an editor registered will still display
		/// in the scene tree, but it will not be editable. Any attributes of the scene object
		/// that do have editors will be editable.</remarks>
		private void RegisterEditors() 
		{
			//component editors
			EditorInfo vectorEd = new EditorInfo();
			vectorEd.EditorFormType = typeof(EditVectorForm);
			vectorEd.NodeColor = Color.DarkGreen;

			EditorInfo materialEd = new EditorInfo();
			materialEd.EditorFormType = typeof(EditMaterialForm);
			materialEd.NodeColor = Color.PaleVioletRed;

			EditorInfo colorEd = new EditorInfo();
			colorEd.EditorFormType = typeof(EditColorForm);
			colorEd.NodeColor = Color.DarkMagenta;

			//shape editors
			EditorInfo sphereEd = new EditorInfo();
			sphereEd.EditorFormType = typeof(EditSphereForm);
			sphereEd.NodeColor = Color.DarkBlue;

			EditorInfo planeEd = new EditorInfo();
			planeEd.EditorFormType=typeof(EditPlaneForm);
			planeEd.NodeColor=Color.Blue;

			EditorInfo meshEd = new EditorInfo();
			meshEd.EditorFormType = typeof(EditPolygonMeshForm);
			meshEd.NodeColor = Color.SteelBlue;

			EditorInfo cylinderEd = new EditorInfo();
			cylinderEd.EditorFormType = typeof(EditCylinderForm);
			cylinderEd.NodeColor = Color.DodgerBlue;

			EditorInfo polygonEd = new EditorInfo();
			polygonEd.EditorFormType = typeof(EditPolygonForm);
			polygonEd.NodeColor = Color.SlateBlue;

			EditorInfo checkerEd = new EditorInfo();
			checkerEd.EditorFormType = typeof(EditCheckerboardForm);
			checkerEd.NodeColor = Color.SkyBlue;

			//misc editors
			EditorInfo lightEd = new EditorInfo();
			lightEd.EditorFormType=typeof(EditLightForm);
			lightEd.NodeColor=Color.Goldenrod;

			EditorInfo propEd = new EditorInfo();
			propEd.EditorFormType=typeof(PropertyEditorForm);
			propEd.NodeColor = Color.Green;

			this.typeEditors[typeof(Vector3f)]=vectorEd;
			this.typeEditors[typeof(Material)]=materialEd;
			this.typeEditors[typeof(Color4f)]=colorEd;

			//this.typeEditors[typeof(Sphere)]=sphereEd;
			this.typeEditors[typeof(Sphere)]=propEd;
			this.typeEditors[typeof(Plane)]=planeEd;
			this.typeEditors[typeof(PolygonMesh)]=meshEd;
			this.typeEditors[typeof(Cylinder)]=cylinderEd;
			this.typeEditors[typeof(Polygon)]=polygonEd;
			this.typeEditors[typeof(Checkerboard)]=checkerEd;

			
			this.typeEditors[typeof(Light)]=lightEd;
		}

		/// <summary>
		/// Returns the System.Type for the form which can edit the given object
		/// </summary>
		/// <param name="obj">The scene object to edit</param>
		/// <returns>A form type that can edit the object, or null if there is no suitable editor</returns>
		private Type GetTypeEditor(object obj) 
		{
			EditorInfo ed = (EditorInfo)typeEditors[obj.GetType()] ;
			if (ed==null)
				return null;
			return ed.EditorFormType;
		}

		/// <summary>
		/// Returns the System.Type for the shape associated with the menu item.
		/// </summary>
		/// <param name="Menu">The menu item clicked on</param>
		/// <returns>A type for the shape, or null if no shape associated with this menu item.</returns>
		private Type GetShapeTypeFromMenuItem(MenuItem Menu) 
		{
			if (Menu==null)
				return null;
			return (Type)menuShapes[Menu];
		}

		/// <summary>
		/// Gets the color associated with the given object
		/// </summary>
		/// <param name="obj">The scene object</param>
		/// <returns>A color assigned to the shape type, or if no assigned color, the default color</returns>
		private Color GetTypeColor(object obj) 
		{
			EditorInfo ed = (EditorInfo)typeEditors[obj.GetType()];
			if (ed==null)
				return this.programSettings.DefaultNodeColor;
			return ed.NodeColor;
		}

		/// <summary>
		/// Loads the default scene
		/// </summary>
		private void LoadDefaultScene() 
		{
			lock(sceneLock) 
			{
				m_scene = new Scene();
			
								
				Sphere s1 = new Sphere(new Vector3f(0.0f,0.0f,2.0f),1.0f);
				s1.Material = new Material(Color.YellowGreen);

				Plane p1 = new Plane(new Vector3f(0.0f,-1.5f,0.0f),new Vector3f(0.0f,1.0f,0.0f));
				p1.Material = new Material(Color.White);

				m_scene.AmbientLight = Color.White;

				m_scene.AddShape(s1);
				m_scene.AddShape(p1);
			
				m_scene.Camera = new Camera(new Vector3f(0.0f,1.0f,-5.0f));
			
				//m_scene.AddLight(new Light(new Vector3f(0.0f,5.0f,2.0f)));
				m_scene.AddLight(new Light(new Vector3f(-5.0f,5.0f,2.0f)));

				m_isDirty = false;

				
			}
            SceneTree_LoadScene(m_scene);
			UpdateTitleBar();
		}

		/// <summary>
		/// Loads a blank scene
		/// </summary>
		private void LoadBlankScene() 
		{
			lock (sceneLock) 
			{
				m_scene = new Scene();

				m_scene.AmbientLight = Color.White;
			
				m_scene.Camera = new Camera(new Vector3f(0.0f,1.0f,-5.0f));
			
				m_scene.AddLight(new Light(new Vector3f(-5.0f,5.0f,-5.0f)));

				m_isDirty = true;
				
			}
			SceneTree_LoadScene(m_scene);
			UpdateTitleBar();
		}

		#endregion

		#region Scene Tree Loading
		/// <summary>
		/// Clears tree list and fills it with a scene's information
		/// </summary>
		/// <param name="S">Scene to show</param>
		/// <remarks>This method loads a new scene by calling SceneTree_AddNode on each scene object</remarks>
		private void SceneTree_LoadScene(Scene S) 
		{
			lock(sceneLock) 
			{
				this.treeViewScene.Nodes.Clear();
				
				//camera
				cameraNode = SceneTree_AddNode(S.Camera,"Camera",null,-1);
				cameraNode.ImageIndex=1;
				cameraNode.SelectedImageIndex=1;
				cameraNode.Expand();

				//lights
				lightsNode = new TreeNode("Lights");
				lightsNode.ImageIndex = 2;
				lightsNode.SelectedImageIndex = 2;
				this.treeViewScene.Nodes.Add(lightsNode);
				foreach(Light l in m_scene.Lights) 
				{
					SceneTree_AddNode(l,"Light",lightsNode,-1);
				}
				lightsNode.Expand();

				//shapes
				this.shapesNode = new TreeNode("Shapes");
				shapesNode.ImageIndex = 4;
				shapesNode.SelectedImageIndex=4;
				this.treeViewScene.Nodes.Add(shapesNode);

				foreach (Shape s in S) 
				{
					SceneTree_AddNode(s,s.GetType().Name,shapesNode,-1);
				}
				shapesNode.Expand();
				
				//scene-global attributes
				TreeNode ambientLightNode = SceneTree_AddNode(S.AmbientLight,"Ambient Light",null,-1);
				TreeNode backgroundColorNode = SceneTree_AddNode(S.BackgroundColor,"Background Color",null,-1);

			}

		}

		/// <summary>
		/// Adds a node (and its children) to the scene tree. 
		/// </summary>
		/// <param name="obj">A scene object</param>
		/// <param name="Name">Name of the object to display in the tree</param>
		/// <param name="Parent">The parent node (can be null)</param>
		/// <param name="index">Index where to insert item (if less than 0, inserts at end)</param>
		/// <returns>The node inserted</returns>
		private TreeNode SceneTree_AddNode(object obj, String Name, TreeNode Parent, int index) 
		{
			//add node for obj
			TreeNode node = new TreeNode(obj.GetType().Name);
			node.ForeColor = GetTypeColor(obj);
			if (Parent!=null) 
			{
				if (index<0)
					Parent.Nodes.Add(node);
				else
					Parent.Nodes.Insert(index,node);
			}
			else 
			{
				if (index<0)
					this.treeViewScene.Nodes.Add(node);
				else
					treeViewScene.Nodes.Insert(index,node);
			}

			SceneNodeInfo info = new SceneNodeInfo();
			info.Data = obj;
			info.Name = Name;
			node.Tag = info;
			
			if (obj is Shape) 
			{
				Shape s = (Shape)obj;
				node.Text = Name + ": "+s.Name;
				node.ImageIndex = 3;
				node.SelectedImageIndex = 3;
			}
			else if (obj is Light) 
			{
				Light l = (Light)obj;
				node.Text = Name + ": "+l.Name;
				node.ImageIndex = 2;
				node.SelectedImageIndex = 2;
			} 
			else if (obj is Camera) 
			{
				node.Text = Name;
				node.ImageIndex=1;
				node.SelectedImageIndex = 1;
			}
			else 
			{
				node.Text = Name+": "+obj.ToString();
			}
			
			
			//get all public properties/fields
			PropertyInfo[] properties = obj.GetType().GetProperties(BindingFlags.Instance | BindingFlags.Public);
			foreach (PropertyInfo prop in properties) 
			{
				if (prop.IsDefined(typeof(EditableAttribute),true)) 
				{
					object val = prop.GetValue(obj,null);
					SceneTree_AddNode(val,prop.Name,node,-1);
				}
				else if (prop.IsDefined(typeof(ViewableAttribute),true)) 
				{
					object val = prop.GetValue(obj,null);
					SceneTree_AddNode(val,prop.Name,node,-1);
				}
			}
			return node;
		}

		#endregion

		/// <summary>
		/// Paint event handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks>All it does is lock the bitmap and draw it.</remarks>
		private void OnPicturePaint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			//draw the bitmap
			lock(bmpLock) 
			{
				e.Graphics.DrawImage(this.bmp,0.0f,0.0f);
			}
		}

		#region Toolbar and Menu Handling
		/// <summary>
		/// Responds to toolbar button clicks
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void toolBar1_ButtonClick(object sender, System.Windows.Forms.ToolBarButtonClickEventArgs e)
		{
			if (e.Button == this.toolBarButtonNewScene) 
			{
				NewScene();
			}
			else if (e.Button == this.toolBarButtonSave) 
			{
				SaveScene();
			} 
			else if (e.Button == this.toolBarButtonOpenScene)
			{
				OpenScene();
			}
			else if (e.Button == this.toolBarButtonStart) 
			{
				StartRendering();
			} 
			else if (e.Button == this.toolBarButtonStop) 
			{
				StopRendering();
			} 
			else if (e.Button == this.toolBarButtonAntiAlias) 
			{
				traceSettings.AntiAlias = this.toolBarButtonAntiAlias.Pushed;
				if (traceSettings.AntiAlias) 
				{
					this.toolBarButtonAntiAlias.ImageIndex++;
					this.comboBoxSubpixels.Enabled=true;
				} 
				else 
				{
					this.toolBarButtonAntiAlias.ImageIndex--;
					this.comboBoxSubpixels.Enabled=false;
				}
			} 
			else if (e.Button == this.toolBarButtonSaveOutput) 
			{
				SaveOutput();
			}
		}

		/// <summary>
		/// Creates a new scene
		/// </summary>
		/// <remarks>Stops rendering if necesary and saves the current scene. Loads a blank scene, not the default one.</remarks>
		private void NewScene() 
		{
			if (bRendering) 
			{
				StopRendering();
			}

			if (this.m_isDirty) 
			{
				this.SaveScene();
			}
			LoadBlankScene();
		}

		/// <summary>
		/// Saves the scene
		/// </summary>
		/// <remarks>If there is no current filename, will call Save As...</remarks>
		private void SaveScene() 
		{
			if (m_scene.Filename==null) 
			{
				SaveSceneAs();
			} 
			else 
			{
				Save();

			}
		}

		/// <summary>
		/// Prompts the user for a filename to save the current scene as
		/// </summary>
		private void SaveSceneAs() 
		{
			SaveFileDialog sfd = new SaveFileDialog();
            sfd.FileName = "New Scene";
			sfd.Filter = programSettings.SceneFilenameFilter;
			
			if (sfd.ShowDialog()==DialogResult.OK) 
			{
				m_scene.Filename = sfd.FileName;
				Save();
			}
		}

		/// <summary>
		/// Saves the current scene to the current scene's filename.
		/// </summary>
		/// <remarks>The filename cannot be null at this point.</remarks>
		private void Save() 
		{
			System.IO.Stream stream = null;
			try 
			{
				stream = System.IO.File.Create(m_scene.Filename);
				System.Runtime.Serialization.Formatters.Soap.SoapFormatter sf = new System.Runtime.Serialization.Formatters.Soap.SoapFormatter();
				lock(sceneLock) 
				{
					sf.Serialize(stream,m_scene);
					stream.Close();
					m_isDirty=false;
				}
				UpdateTitleBar();
			}
			catch (Exception ex) 
			{
				MessageBox.Show(ex.Message);
			}
			finally 
			{
				stream.Close();
			}
			
		}

		/// <summary>
		/// Prompts the user to select a scene file to open
		/// </summary>
		private void OpenScene() 
		{
			System.IO.Stream stream = null;
			try 
			{
				if (bRendering)
					StopRendering();
				if (m_isDirty) 
				{
					DialogResult result =MessageBox.Show("Would you like to save the current scene?","Save scene?",MessageBoxButtons.YesNoCancel);
					if (result==DialogResult.Yes) 
					{
						SaveScene();
					} else if (result==DialogResult.Cancel)
						return;
				}
				OpenFileDialog ofd = new OpenFileDialog();
				ofd.Filter = programSettings.SceneFilenameFilter;
				if (ofd.ShowDialog()==DialogResult.OK) 
				{
				
					stream = System.IO.File.OpenRead(ofd.FileName);
					System.Runtime.Serialization.Formatters.Soap.SoapFormatter sf = new System.Runtime.Serialization.Formatters.Soap.SoapFormatter();
					lock(sceneLock) 
					{
						m_scene = (Scene)sf.Deserialize(stream);
						m_scene.Filename = ofd.FileName;
						m_isDirty = false;
						this.SceneTree_LoadScene(m_scene);
						UpdateTitleBar();
					}
				}
			}
			catch (Exception ex) 
			{
				MessageBox.Show(ex.Message);
			}
			finally 
			{
				if (stream!=null)
					stream.Close();
			}
		}

		/// <summary>
		/// Exports the current bitmap to an image file
		/// </summary>
		/// <remarks>Uses the ImageFormatInfo structure to construct the file filters and decide 
		/// which format to save as</remarks>
		private void SaveOutput() 
		{
			SaveFileDialog sfd = new SaveFileDialog();
			sfd.Filter =  programSettings.ImageFilters;
			sfd.FileName = "New Snapshot";

			if (sfd.ShowDialog()==DialogResult.OK) 
			{
				lock(bmpLock) 
				{
					bmp.Save(sfd.FileName, imageFormats[sfd.FilterIndex-1].Format);
				}
			}
			
		}

		/// <summary>
		/// Event handler for when the maximum trace depth changes
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnDepthChanged(object sender, System.EventArgs e)
		{
			String str = this.comboBoxDepth.Text;
			try 
			{
				int val = int.Parse(str);
				this.traceSettings.TraceLevels = val;
			}
			catch 
			{
				MessageBox.Show("You must enter a valid integer value for the depth");
				this.comboBoxDepth.Focus();
			}
		}

		/// <summary>
		/// Event handler for when the number of subpixels changes
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnSubpixelsChanged(object sender, System.EventArgs e)
		{
			String str = this.comboBoxSubpixels.Text;
			try 
			{
				int val = int.Parse(str);
				this.traceSettings.SubPixels = val;
			}
			catch 
			{
				MessageBox.Show("You must enter a valid integer value for the number of subpixels");
				this.comboBoxSubpixels.Focus();
			}
		}

		/// <summary>
		/// Event handler for when the output image size changes
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnImageSizeSelectionChanged(object sender, System.EventArgs e)
		{
			ImageSize[] sizes = (ImageSize[])this.comboBoxImageSize.Tag;
			ImageSize size = sizes[this.comboBoxImageSize.SelectedIndex];
			this.traceSettings.ImageWidth = size.Width;
			this.traceSettings.ImageHeight = size.Height;
		}

		/// <summary>
		/// Responds to File | Exit
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void menuItemFileExit_Click(object sender, System.EventArgs e)
		{
			Shutdown();
			Application.Exit();
		}

		/// <summary>
		/// Responds to add light menu item
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnAddLight(object sender, System.EventArgs e)
		{
			AddNewLight();
		}

		#endregion

		#region Rendering Procedures

		/// <summary>
		/// Starts the rendering procedure on the current scene
		/// </summary>
		/// <remarks>Stops any previous rendering, if necessary. Creates a new background thread with which to render.
		/// </remarks>
		private void StartRendering() 
		{
			if (bRendering)
				StopRendering();
			
			renderThread = new Thread(new ThreadStart(RenderProc));
			renderThread.IsBackground = true;
			renderThread.Priority = programSettings.RenderPriority;
			
			RenderStarted(this,EventArgs.Empty);

			tracer = new Tracer(traceSettings,m_scene);

			renderThread.Start();
		}


		/// <summary>
		/// Stops the rendering process
		/// </summary>
		/// <remarks>This procedure first attempts a clean shutdown. If rendering termination signal is not received in
		/// 10 seconds, it forcibly terminates the thread</remarks>
		private void StopRendering() 
		{
			bRendering = false;

			//wait until thread stops
			bool graceful = renderProcDone.WaitOne(10000,false);
			if (!graceful) 
			{
				renderThread.Abort();
				RenderFinished(this,EventArgs.Empty);
			}
		}

		/// <summary>
		/// The rendering procedure
		/// </summary>
		/// <remarks>
		/// Rendering is accomplished in two stages:
		/// 1) a quick and dirty "preview" version.
		/// If antialiasing is not enabled then:
		/// 2a) each pixel is calculated.
		/// If antialiasing is enabled then:
		/// 2b) subpixels are calculated and averaged for each pixel.
		/// 
		/// A uniform subpixel weighting is used.
		/// </remarks>
		private void RenderProc() 
		{
			
			this.renderProcDone.Reset();
		
			int width = this.traceSettings.ImageWidth;
			int height = this.traceSettings.ImageHeight;
			
			this.bmp = new Bitmap(width,height);
			Stats.NumberOfPixels = width*height;

	
			//do a quick first pass to get general shape of things
			lock(sceneLock) 
			{
				this.tracer.DoPreRenderCalculations();//helps speed things up
				lock(bmpLock) 
				{
					for (int i=0;i<height && bRendering;i++) 
					{
						for (int j=0;j<width && bRendering;j++) 
						{
							//get ray for this pixel position
							Ray ray = tracer.GetRayFromPixel(j,i);
							Color c = tracer.GetQuickColorOfRay(ray);
							bmp.SetPixel(j,i,c);
						}
					}
				}
			}
			this.pictureBox1.Refresh();
			
			//setup up statistics
			Stats.Reset();
			Stats.NumberOfPixels = width*height;
			Stats.TotalLines = height;
			Stats.StartTime = DateTime.Now;

			if (this.traceSettings.AntiAlias) 
			{
				//compute values for subpixels
				int sub=(int)Math.Sqrt(traceSettings.SubPixels);
				int numSubPixels = traceSettings.SubPixels;
				for (int i=0;i<height && bRendering;i++) 
				{
					lock(sceneLock) 
					{
						lock (bmpLock) 
						{
							for (int j=0;j<width && bRendering;j++) 
							{
								Color4f totalColor = new Color4f();

								//get values of subpixels and average them
								for (int si=0;si<sub;si++) 
								{
									for (int sj=0;sj<sub;sj++) 
									{
										Ray ray = tracer.GetRayFromPixelAA(j,i,sj,si);
										Color4f c = tracer.GetColorOfRay(ray);
										totalColor+=c/(float)numSubPixels;
									}
								}
		
								bmp.SetPixel(j,i,totalColor);
							}
						}
					}
					//update every line
					this.pictureBox1.Refresh();
					Stats.NumberOfLinesTraced++;
					Stats.FinishTime=DateTime.Now;
					
					this.DisplayStats();
					this.UpdateTitleBar();
					
				}
			} 
			else 
			{
				//compute values for pixels
				for (int i=0;i<height && bRendering;i++) 
				{
					lock(sceneLock) 
					{
						lock(bmpLock) 
						{
							for (int j=0;j<width && bRendering;j++) 
							{
								//get ray for this pixel position
								Ray ray = tracer.GetRayFromPixel(j,i);

								//get its color
								Color color = tracer.GetColorOfRay(ray);
								bmp.SetPixel(j,i,color);
	
							}
						}
					}
					this.pictureBox1.Refresh();
					Stats.NumberOfLinesTraced++;
					Stats.FinishTime=DateTime.Now;
					
					this.DisplayStats();
					this.UpdateTitleBar();
				}

			}
		
			Stats.FinishTime=DateTime.Now;
	
			//fire event
			this.renderProcDone.Set();

			RenderFinished(this,EventArgs.Empty);

		
		}

		private void MainForm_RenderStarted(object sender, EventArgs e)
		{
			this.toolBarButtonStop.Enabled=true;
			this.toolBarButtonStart.Enabled=false;
			this.toolBarButtonAntiAlias.Enabled=false;
			this.toolBarButtonNewScene.Enabled=false;
			this.toolBarButtonOpenScene.Enabled=false;
			this.toolBarButtonSaveOutput.Enabled=false;
			this.comboBoxSubpixels.Enabled=false;
			this.comboBoxImageSize.Enabled=false;
			this.comboBoxDepth.Enabled=false;
			this.treeViewScene.Enabled=false;
			this.mainMenuItemShapes.Enabled=this.mainMenuItemLights.Enabled=this.mainMenuItemCamera.Enabled=false;
			this.menuItemNewScene.Enabled=this.menuItemOpenScene.Enabled=false;
			
			this.statusBarPanelGeneral.Text = "Rendering...";
			this.Cursor = Cursors.AppStarting;
			bRendering = true;
		}

		private void MainForm_RenderFinished(object sender, EventArgs e)
		{
			if (this.IsDisposed==false) 
			{


				this.toolBarButtonStop.Enabled=false;
				this.toolBarButtonStart.Enabled=true;
				this.toolBarButtonAntiAlias.Enabled=true;
				this.toolBarButtonNewScene.Enabled=true;
				this.toolBarButtonOpenScene.Enabled=true;
				this.toolBarButtonSaveOutput.Enabled=true;
				this.comboBoxImageSize.Enabled=true;
				this.comboBoxSubpixels.Enabled = this.traceSettings.AntiAlias;
				this.comboBoxDepth.Enabled=true;
				this.treeViewScene.Enabled=true;
				this.mainMenuItemShapes.Enabled=this.mainMenuItemLights.Enabled=this.mainMenuItemCamera.Enabled=true;
				this.menuItemNewScene.Enabled=this.menuItemOpenScene.Enabled=true;

				this.bRendering=false;
				this.Cursor = Cursors.Default;
				this.statusBarPanelGeneral.Text = "Ready";
				
				//this.tabControl1.SelectedIndex=1;
				DisplayFullStats();
				DisplayStats();
				UpdateTitleBar();
			}
		}

		#endregion

		#region Statistical Display, Status bar, and Title bar
		/// <summary>
		/// Displays rendering statistics in the status bar
		/// </summary>
		private void DisplayStats() 
		{
			this.statusBarPanelElapsedTime.Text = String.Format("{0}h {1}m {2}s",Stats.ElapsedTime.Hours,Stats.ElapsedTime.Minutes,Stats.ElapsedTime.Seconds);
			this.statusBarPanelPixelsPerSecond.Text = String.Format("{0:n2} pixels/sec",(float)Stats.NumberOfPixels/Stats.ElapsedTime.Seconds);
			this.statusBarPanelRaysPerSecond.Text = String.Format("{0:n2} rays/sec",Stats.RaysPerSecond);
			this.statusBarPanelRaysTraced.Text = String.Format("{0:n0} rays total",Stats.NumberOfRaysTraced);
		}


		/// <summary>
		/// Displays full stats in the stats tab
		/// </summary>
		private void DisplayFullStats() 
		{
			Font regularFont = new Font("Times New Roman",12,FontStyle.Regular);
			Font headingFont = new Font("Times New Roman", 12,FontStyle.Bold);
			String ElapsedTime = String.Format("{0}h {1}m {2}s",Stats.ElapsedTime.Hours,Stats.ElapsedTime.Minutes,Stats.ElapsedTime.Seconds);
			//this.richTextBoxStats.Text = "\n";
			//this.richTextBoxStats.SelectionFont = headingFont;
			this.richTextBoxStats.AppendText("\nElapsed Time: ");
			//this.richTextBoxStats.SelectionFont = regularFont;
			this.richTextBoxStats.AppendText(ElapsedTime);
		}

		/// <summary>
		/// Updates the title bar to include the current file, dirty flag, and rendering progress
		/// </summary>
		private void UpdateTitleBar() 
		{
			System.Text.StringBuilder sb= new System.Text.StringBuilder();
			sb.Append("BRayTracer - ");
			if (m_scene.Filename==null)
				sb.Append("Untitled Scene");
			else 
				sb.Append(m_scene.Filename);

			if (m_isDirty) 
			{
				sb.Append(" *");
			}

			if (bRendering) 
			{
				float percent = (((float)Stats.NumberOfLinesTraced)/((float)Stats.TotalLines) * 100.0f);
				sb.Append(" (Rendering - ");
				sb.Append(percent.ToString("F1"));
				sb.Append("% complete)");
			}

			this.Text = sb.ToString();

		}
		#endregion

		#region Adding, Editing, Deleting Scene Objects
		/// <summary>
		/// Responds to double clicks in the scene tree
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnSceneTreeDoubleClick(object sender, System.EventArgs e)
		{
			TreeNode node = this.treeViewScene.SelectedNode;
			if (node==null)
				return;

			SceneNodeInfo info = (SceneNodeInfo)node.Tag;
			if (info!=null) 
			{
				if (info.Data!=null) 
				{
					EditObject(info, node);
				}
			}
		}

		/// <summary>
		/// Responds to mouse clicks in the scene tree
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks>If the right mouse button is pressed, a context menu is displayed
		/// which is specific to the current node.</remarks>
		private void OnSceneTree_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if (e.Button == MouseButtons.Right) 
			{
				TreeNode node = this.treeViewScene.GetNodeAt(e.X,e.Y);
				treeViewScene.SelectedNode = node;
				if (node==null)
					return;

				if (node == shapesNode) 
				{
					this.contextMenuShapeCollection.Show(this.treeViewScene,new Point(e.X,e.Y));
				} 
				else if (node == lightsNode) 
				{
					this.contextMenuLights.Show(this.treeViewScene,new Point(e.X,e.Y));
				} 
				else if (node == cameraNode) 
				{
					this.menuItemRemoveSceneObject.Enabled=false;
					this.menuItemEditSceneObject.Enabled=true;
					this.contextMenuSceneObject.Show(this.treeViewScene,new Point(e.X,e.Y));
				} 
				else 
				{
					SceneNodeInfo info = (SceneNodeInfo)node.Tag;
					if (info!=null)
					{
						if (info.Data is Shape || info.Data is Light) 
						{
							//must be a shape--enable removal
							this.menuItemRemoveSceneObject.Enabled=true;
						} 
						else 
						{
							//must be a shape component--can't remove these
							this.menuItemRemoveSceneObject.Enabled=false;
						}
						//see if there is an editor for this object
						object ed = GetTypeEditor(info.Data);
						if (ed!=null) 
						{
							//we have an editor, so enable editing
							this.menuItemEditSceneObject.Enabled=true;

						} 
						else 
						{
							this.menuItemEditSceneObject.Enabled=false;
						}
						this.contextMenuSceneObject.Show(this.treeViewScene,new Point(e.X,e.Y));

					} 
					else 
					{
						//do nothing
					}
				}
			}
		}

		/// <summary>
		/// Edits the current scene object
		/// </summary>
		/// <param name="Info"></param>
		/// <param name="node"></param>
		/// <returns>True if the scene object was succesfully edited, false if the user canceled the operation</returns>
		private bool EditObject(SceneNodeInfo Info, TreeNode node)
		{
		
			Type formType = (Type)GetTypeEditor(Info.Data);

			if (formType!=null) 
			{
				IEditor form = (IEditor)Activator.CreateInstance(typeof(PropertyEditorForm));
				form.EditableObject = Info.Data;
				if (form.ShowDialog()==DialogResult.OK) 
				{
					m_isDirty=true;
					UpdateTitleBar();
				
					int idx = node.Index;
					
					TreeNode parent = node.Parent;
					
					TreeNode newNode = this.SceneTree_AddNode(Info.Data,Info.Name,parent,idx);
					node.Remove();
					newNode.Expand();
					
					treeViewScene.SelectedNode=newNode;
					return true;
				}
			}
            return false;
		}



		/// <summary>
		/// Adds a new light
		/// </summary>
		private void AddNewLight() 
		{
			Light light = new Light();
			Type edForm = (Type)GetTypeEditor(light);
			if (edForm!=null) 
			{
				IEditor editor = (IEditor)Activator.CreateInstance(edForm);
				if (editor!=null) 
				{
					editor.EditableObject = light;
					if (editor.ShowDialog()==DialogResult.OK) 
					{
						m_scene.AddLight(light);
						TreeNode node = SceneTree_AddNode(light,light.GetType().Name,this.lightsNode,-1);
						node.Expand();
					}
				}
			}


		}

		/// <summary>
		/// Event handler for adding a new light
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnAddNewShape(object sender, EventArgs e) 
		{
			//see what type of shape is associated with the menu clicked
			Type shapeType = GetShapeTypeFromMenuItem((MenuItem)sender);
			EditorInfo ed = null;

			//check to see if there is an editor for this type 
			
			if (shapeType!=null)
				ed = (EditorInfo)typeEditors[shapeType];
			if (ed==null) 
			{
				MessageBox.Show("There is no registered editor for that type of shape","BRayTracer");
				return;
			}

			Shape s = (Shape)Activator.CreateInstance(shapeType);
			if (s!=null) 
			{
				IEditor editor = (IEditor)Activator.CreateInstance(typeof(PropertyEditorForm));
				if (editor!=null) 
				{
					editor.EditableObject = s;
					if (editor.ShowDialog()==DialogResult.OK) 
					{
						m_scene.AddShape(s);
						TreeNode node = SceneTree_AddNode(s,s.GetType().Name,
							this.shapesNode,-1);
						node.Expand();
					}
				}
			}
		}
		
		/// <summary>
		/// Event handler for editing the current scene object
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnEditSceneObject(object sender, EventArgs e)
		{
			TreeNode node = this.treeViewScene.SelectedNode;
			SceneNodeInfo info = (SceneNodeInfo)node.Tag;
			if (GetTypeEditor(info.Data)!=null) 
			{
				lock(sceneLock) 
				{
					EditObject(info, node);
				}
			}
		}

		/// <summary>
		/// Event handler for removing the current scene object
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OnRemoveSceneObject(object sender, EventArgs e)
		{
			TreeNode node = this.treeViewScene.SelectedNode;
			SceneNodeInfo info = (SceneNodeInfo)node.Tag;
			if (info.Data is Shape) 
			{
				lock(sceneLock) 
				{
					m_scene.RemoveShape((Shape)info.Data);
					node.Remove();
					m_isDirty=true;
					UpdateTitleBar();
				}
			} 
			else if (info.Data is Light) 
			{
				lock(sceneLock) 
				{
					m_scene.RemoveLight((Light)info.Data);
					node.Remove();
					m_isDirty=true;
					UpdateTitleBar();
				}
			}

		}

		#endregion

		#region Shutdown Code
		/// <summary>
		/// Event handler for the form closing
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks>Calls Shutdown() to perform any cleaning up code</remarks>
		private void OnMainFormClosing(object sender, System.ComponentModel.CancelEventArgs e)
		{
			Shutdown();
		}

		/// <summary>
		/// Stops rendering and saves the scene
		/// </summary>
		private void Shutdown() 
		{
			if (bRendering)
				StopRendering();

			if (m_isDirty) 
			{
				this.SaveScene();
			}
		}
		#endregion

		private void menuItemAbout_Click(object sender, System.EventArgs e)
		{
			AboutForm form = new AboutForm();
			form.ShowDialog();
		}

		private void OnRenderPriorityChanged(object sender, System.EventArgs e)
		{
			//uncheck all

			this.menuItemRenderPriorityHigh.Checked = this.menuItemRenderPriorityAboveNormal.Checked =
				this.menuItemRenderPriorityNormal.Checked = this.menuItemRenderPriorityBelowNormal.Checked =
				this.menuItemRenderPriorityLow.Checked = false;

			if (sender==this.menuItemRenderPriorityHigh) 
			{
				this.programSettings.RenderPriority = ThreadPriority.Highest;
				this.menuItemRenderPriorityHigh.Checked = true;
			} 
			else if (sender==this.menuItemRenderPriorityAboveNormal) 
			{
				this.programSettings.RenderPriority = ThreadPriority.AboveNormal;
				this.menuItemRenderPriorityAboveNormal.Checked = true;
			} 
			else if (sender==this.menuItemRenderPriorityNormal) 
			{
				this.programSettings.RenderPriority = ThreadPriority.Normal;
				this.menuItemRenderPriorityNormal.Checked = true;
			} 
			else if (sender==this.menuItemRenderPriorityBelowNormal) 
			{
				this.programSettings.RenderPriority = ThreadPriority.BelowNormal;
				this.menuItemRenderPriorityBelowNormal.Checked = true;
			} 
			else if (sender==this.menuItemRenderPriorityLow) 
			{
				this.programSettings.RenderPriority = ThreadPriority.Lowest;
				this.menuItemRenderPriorityLow.Checked = true;
			}

			if (this.bRendering) 
			{
				this.renderThread.Priority = this.programSettings.RenderPriority;

			}
		}

		private void menuItemRenderStart_Click(object sender, System.EventArgs e)
		{
			StartRendering();

		
		}

		private void menuItemRenderStop_Click(object sender, System.EventArgs e)
		{
			StopRendering();
		
		}



	}
}
