/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/21/2004
 *
 * Abstract:
 *     A composite UI control to edit colors
 *
 * Revision History:
 *		12/20/2004 - Added to VSS
 *		12/21/2004 - Ben Watson - added XML comments
 *		
 * 
 *
 *--*/

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

using BenWatson.BRayTracer;

namespace BenWatson.BRayTracer.Editors
{
	/// <summary>
	/// ColorEditorControl is a composite UI control that allows for editing of
	/// Color4f objects.
	/// </summary>
	/// <remarks>Edit is accomplished with the System Color dialog.
	/// Currently, there is a disconnect: the internal format used here is in floating-point
	/// but the system uses integers from 0-255.</remarks>
	public class ColorEditorControl : System.Windows.Forms.UserControl
	{
		#region UI elements
		private System.Windows.Forms.Label labelColorLabel;
		private System.Windows.Forms.Button buttonEdit;
		private System.Windows.Forms.ColorDialog colorDialog;
		/// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		#endregion
		
		/// <summary>
		/// Copy of the original color
		/// </summary>
		private Color4f m_color = null;
		
		/// <summary>
		/// Notifies listeners when the color has changed
		/// </summary>
		public event EventHandler ColorChanged;

		/// <summary>
		/// Initializes a new ColorEditorControl
		/// </summary>
		public ColorEditorControl()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			// TODO: Add any initialization after the InitializeComponent call

		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.labelColorLabel = new System.Windows.Forms.Label();
			this.buttonEdit = new System.Windows.Forms.Button();
			this.colorDialog = new System.Windows.Forms.ColorDialog();
			this.SuspendLayout();
			// 
			// labelColorLabel
			// 
			this.labelColorLabel.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.labelColorLabel.Location = new System.Drawing.Point(8, 8);
			this.labelColorLabel.Name = "labelColorLabel";
			this.labelColorLabel.Size = new System.Drawing.Size(208, 23);
			this.labelColorLabel.TabIndex = 0;
			this.labelColorLabel.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			// 
			// buttonEdit
			// 
			this.buttonEdit.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.buttonEdit.Location = new System.Drawing.Point(224, 8);
			this.buttonEdit.Name = "buttonEdit";
			this.buttonEdit.Size = new System.Drawing.Size(48, 23);
			this.buttonEdit.TabIndex = 1;
			this.buttonEdit.Text = "&Edit...";
			this.buttonEdit.Click += new System.EventHandler(this.buttonEdit_Click);
			// 
			// colorDialog
			// 
			this.colorDialog.AnyColor = true;
			this.colorDialog.FullOpen = true;
			this.colorDialog.SolidColorOnly = true;
			// 
			// ColorEditorControl
			// 
			this.Controls.Add(this.buttonEdit);
			this.Controls.Add(this.labelColorLabel);
			this.Name = "ColorEditorControl";
			this.Size = new System.Drawing.Size(280, 40);
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// Gets and sets the color for this control
		/// </summary>
		/// <remarks>Setting the color makes a copy.</remarks>
		public Color4f Color 
		{
			get 
			{
				return m_color;
			}
			set 
			{
				if (value!=null) {
					m_color = (Color4f)(value.Clone());
					UpdateView();
				}
			}
		}

		/// <summary>
		/// Updates the background color and text of the color display.
		/// </summary>
		private void UpdateView() 
		{
			if (m_color!=null)
			{
				this.labelColorLabel.Text = m_color.ToString();
				this.labelColorLabel.BackColor = m_color.SysColor;
			}
		}

		/// <summary>
		/// Event handler for the edit color button
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <remarks>Displays the system color chooser dialog</remarks>
		private void buttonEdit_Click(object sender, System.EventArgs e)
		{
			this.colorDialog.Color = this.m_color.SysColor;
			if (colorDialog.ShowDialog() == DialogResult.OK) 
			{
				m_color = colorDialog.Color;
				UpdateView();

				if (ColorChanged!=null)
					ColorChanged(this,EventArgs.Empty);
				
			}
		}
	}
}
