/*++
 * Copyright 2004 Ben Watson. This code may be used for non-commercial purposes only. You must
 * credit the author.
 *
 * Module-Name:
 *     BRayTracer
 *
 * Author:
 *     Ben Watson (dev@benwatson.org) 12/20/2004
 *
 * Abstract:
 *     Color4f class - represents colors in 4-value floating point (R, G, B, A) with
 * range 0.0 to 1.0.
 *
 * Revision History:
 * 
 * 12/20/2004 - Added existing file to source control
 * 12/20/2004 - Updated all XML comments
 *
 *--*/

using System;
using System.ComponentModel;
using System.Drawing;

namespace BenWatson.BRayTracer
{
	/// <summary>
	/// Describes a color in floating-point format
	/// </summary>
	/// <remarks>Each component (R, G, B) can have values from 0.0 to 1.0. Values outside this range can be clamped.
	/// There are two ways to manipulate color values: static operators and instance operations.
	/// Static operators create a new Color4f object, while the instance operations operate on the this value.</remarks>
	[Serializable, TypeConverter(typeof(ExpandableObjectConverter))]
	public class Color4f : ICloneable
	{
		/// <summary>
		/// Red component
		/// </summary>
		public float Red;
		/// <summary>
		/// Green component
		/// </summary>
		public float Green;
		/// <summary>
		/// Blue component
		/// </summary>
		public float Blue;
		/// <summary>
		/// Alpha channel component
		/// </summary>
		public float Alpha;
		
		
		#region Properties
		/// <summary>
		/// Gets or sets the color in the system format
		/// </summary>
		/// <remarks>Clamps the color before getting it. When setting, it assumes that Color's components
		/// are in the range 0 to 255.</remarks>
		[Category("General"), Description("Color")]
		public Color SysColor
		{
			get 
			{
				Clamp();
				
				return Color.FromArgb((int)(Alpha * 255.0f),(int)(Red*255.0),(int)(Green*255.0),(int)(Blue*255.0));
			}
			set 
			{
				Alpha = (float)(value.A) / 255.0f;
				Red = (float)(value.R) / 255.0f;
				Green = (float)(value.G) / 255.0f;
				Blue = (float)(value.B) / 255.0f;
			}
		}
		#endregion

		#region Constructors
		/// <summary>
		/// Initializes a new color to black
		/// </summary>
		/// <remarks>Alpha is set to 1.0</remarks>
		public Color4f() 
		{
			Red=Green=Blue=0.0f;
			Alpha=1.0f;
		}
		/// <summary>
		/// Initializes a new color from the given values.
		/// </summary>
		/// <param name="Red">Red component</param>
		/// <param name="Green">Green component</param>
		/// <param name="Blue">Blue component</param>
		/// <remarks>Alpha is set to 1.0</remarks>
		public Color4f(float Red, float Green, float Blue)
		{
			this.Red = Red;
			this.Green = Green;
			this.Blue = Blue;
			this.Alpha = 1.0f;
		}

		/// <summary>
		/// Initializes a new color from given values
		/// </summary>
		/// <param name="Red">Red component</param>
		/// <param name="Green">Green component</param>
		/// <param name="Blue">Blue component</param>
		/// <param name="Alpha">Alpha component</param>
		public Color4f(float Red, float Green, float Blue, float Alpha)
		{
			this.Red = Red;
			this.Green = Green;
			this.Blue = Blue;
			this.Alpha = Alpha;
		}

		#endregion

		#region Static Operators
		/*
		/// <summary>
		/// Determines if two colors are equal (including alpha channel)
		/// </summary>
		/// <param name="C1">First color</param>
		/// <param name="C2">Second color</param>
		/// <returns>True if colors are equal, false otherwise</returns>
		public static bool operator==(Color4f C1, Color4f C2) 
		{
			return ((C1.Red==C2.Red) && (C1.Green==C2.Green) && (C1.Blue == C2.Blue) && (C1.Alpha == C2.Alpha));
		}

		/// <summary>
		/// Determines if two colors are not equal (including alpha channel)
		/// </summary>
		/// <param name="C1">First color</param>
		/// <param name="C2">Second color</param>
		/// <returns>True if colors are not equal, false otherwise</returns>
		public static bool operator!=(Color4f C1, Color4f C2) 
		{
			return !((C1.Red==C2.Red) && (C1.Green==C2.Green) && (C1.Blue == C2.Blue) && (C1.Alpha == C2.Alpha));
		}*/

		/// <summary>
		/// Adds components of colors together.
		/// Does not modify alpha channel.
		/// </summary>
		/// <param name="C1">First color</param>
		/// <param name="C2">Second color</param>
		/// <returns>A new color representing combination of source colors</returns>
		public static Color4f operator+(Color4f C1, Color4f C2) 
		{
			return new Color4f(C1.Red+C2.Red,C1.Green+C2.Green,C1.Blue+C2.Blue);
		}

		/// <summary>
		/// Applies the given intensity to a color.
		/// Does not modify alpha channel
		/// </summary>
		/// <param name="C1">Color</param>
		/// <param name="Intensity">Intensity</param>
		/// <returns>A new color with constant value added to each component</returns>
		public static Color4f operator+(Color4f C1, float Intensity) 
		{
			Color4f newColor = (Color4f)C1.Clone();
			newColor.Red += Intensity;
			newColor.Green += Intensity;
			newColor.Blue += Intensity;

			return newColor;
		}

		/// <summary>
		/// Substracts a color from another color
		/// </summary>
		/// <param name="C1">First color</param>
		/// <param name="C2">Color to subtract</param>
		/// <returns>A new color representing C1 - C2</returns>
		public static Color4f operator-(Color4f C1, Color4f C2) 
		{
			return new Color4f(C1.Red-C2.Red,C1.Green-C2.Green,C1.Blue-C2.Blue,C1.Alpha-C2.Alpha);
		}

		/// <summary>
		/// Substracts the given intensity to a color.
		/// Does not modify alpha channel
		/// </summary>
		/// <param name="C1">Color</param>
		/// <param name="Intensity">Intensity</param>
		/// <returns>A new color with constant value added to each component</returns>
		public static Color4f operator-(Color4f C1, float Intensity) 
		{
			Color4f newColor = (Color4f)C1.Clone();
			newColor.Red -= Intensity;
			newColor.Green -= Intensity;
			newColor.Blue -= Intensity;

			return newColor;
		}

		/// <summary>
		/// Multiplies each color component by the given Scale
		/// Does not mofify alpha channel.
		/// </summary>
		/// <param name="C">The Color to modify</param>
		/// <param name="Scale">Amount to scale each component</param>
		/// <returns>A new color, scaled</returns>
		public static Color4f operator*(Color4f C, float Scale) 
		{
			Color4f newColor = (Color4f)C.Clone();
			newColor.Red*=Scale;
			newColor.Green*=Scale;
			newColor.Blue*=Scale;

			return newColor;
		}

		/// <summary>
		/// Multiplies a color's components by another color's components
		/// </summary>
		/// <param name="C">Source color</param>
		/// <param name="ColorScale">Scale color</param>
		/// <returns>A new color</returns>
		public static Color4f operator*(Color4f C, Color4f ColorScale) 
		{
			Color4f newColor = (Color4f)C.Clone();
            newColor.Red *= ColorScale.Red;
			newColor.Green *= ColorScale.Green;
			newColor.Blue *= ColorScale.Blue;
			return newColor;
		}

		/// <summary>
		/// Divides each color component by the given Scale
		/// Does not mofify alpha channel.
		/// </summary>
		/// <param name="C">The Color to modify</param>
		/// <param name="Scale">Amount to scale each component</param>
		/// <returns>A new color, scaled</returns>
		public static Color4f operator/(Color4f C, float Scale) 
		{
			Color4f newColor = (Color4f)C.Clone();
			newColor.Red/=Scale;
			newColor.Green/=Scale;
			newColor.Blue/=Scale;

			return newColor;
		}

		/// <summary>
		/// Implicity converts a System.Drawing.Color to a Color4f
		/// </summary>
		/// <param name="Color">System color</param>
		/// <returns>a Color4f</returns>
		public static implicit operator Color4f(System.Drawing.Color Color) 
		{
			Color4f c = new Color4f();
			c.SysColor = Color;
			return c;
		}

		/// <summary>
		/// Implicitly converts a Color4f to a System.Drawing.Color
		/// </summary>
		/// <param name="Color">Color4f value to convert</param>
		/// <returns>Equivalent System.Drawing.Color</returns>
		public static implicit operator System.Drawing.Color(Color4f Color) 
		{
			return Color.SysColor;
		}

		
		#endregion

		#region Instance Operators
		/// <summary>
		/// Adds the argument's value to this color
		/// </summary>
		/// <param name="Color">Color to add</param>
		public void Add(Color4f Color) 
		{
			Red+=Color.Red;
			Green+=Color.Green;
			Blue+=Color.Blue;
		}

		/// <summary>
		/// Adds a constant value to this color
		/// </summary>
		/// <param name="Val">The constant value to add</param>
		public void Add(float Val) 
		{
			Red+=Val;
			Green+=Val;
			Blue+=Val;
		}
		/// <summary>
		/// Subtracts a color value from this color
		/// </summary>
		/// <param name="Color">The color to subtract</param>
		public void Subtract(Color4f Color) 
		{
			Red-=Color.Red;
			Green-=Color.Green;
			Blue-=Color.Blue;
		}

		/// <summary>
		/// Multiplies this color by the given scale
		/// </summary>
		/// <param name="Scale">The scale to multiply this color by</param>
		public void Multiply(float Scale) 
		{
			Red*=Scale;
			Green*=Scale;
			Blue*=Scale;
		}

		/// <summary>
		/// Multiplies this color's components by the same components in the given Color
		/// </summary>
		/// <param name="Color">The color to scale this one by</param>
		public void Multiply(Color4f Color) 
		{
			Red*=Color.Red;
			Green*=Color.Green;
			Blue*=Color.Blue;
		}

		/// <summary>
		/// Divides this color's components by the scale
		/// </summary>
		/// <param name="Scale">The given scale</param>
		public void Divide(float Scale) 
		{
			Red/=Scale;
			Green/=Scale;
			Blue/=Scale;
		}
		#endregion

		/// <summary>
		/// Clamps all values to 0.0f - 1.0f range
		/// </summary>
		public void Clamp() 
		{
			if (Red>1.0f)
				Red = 1.0f;
			else if (Red<0.0f)
				Red=0.0f;

			if (Blue>1.0f)
				Blue = 1.0f;
			else if (Blue < 0.0f)
				Blue = 0.0f;

			if (Green > 1.0f)
				Green = 1.0f;
			else if (Green < 0.0f)
				Green = 0.0f;

			if (Alpha > 1.0f)
				Alpha = 1.0f;
			else if (Alpha < 0.0f)
				Alpha = 0.0f;
		}

		/// <summary>
		/// Clamps all values to to the specified range
		/// </summary>
		/// <param name="Min">Minimum value to clamp to</param>
		/// <param name="Max">Maximum value to clamp to</param>
		public void Clamp(float Min, float Max) 
		{
			if (Red>Max)
				Red = Max;
			else if (Red<Min)
				Red=Min;

			if (Blue>Max)
				Blue = Max;
			else if (Blue < Min)
				Blue = Min;

			if (Green > Max)
				Green = Max;
			else if (Green < Min)
				Green = Min;

			if (Alpha > Max)
				Alpha = Max;
			else if (Alpha < Min)
				Alpha = Min;
		}

		#region Object overrides

		/// <summary>
		/// Determines if this Color4f has the same value as the argument
		/// </summary>
		/// <param name="obj">Object to compare for equality</param>
		/// <returns>True if both colors have same values; false otherwise</returns>
		public override bool Equals(object obj)
		{
			if (obj is Color4f) 
			{
				Color4f c=(Color4f)obj;
				return (this==c);
			}
			return false;
		}

		/// <summary>
		/// Returns the default hash code
		/// </summary>
		/// <returns>Default hashcode</returns>
		public override int GetHashCode()
		{
			return base.GetHashCode ();
		}


		/// <summary>
		/// Gets a string version of this color
		/// </summary>
		/// <returns>A string representing this color</returns>
		public override string ToString()
		{
			return String.Format("[R:{0:F2} G:{1:F2} B:{2:F2} A:{3:F2}]",Red,Green,Blue,Alpha);
		}
		#endregion

		#region ICloneable Members

		/// <summary>
		/// Makes a copy of this color object
		/// </summary>
		/// <returns>A copy of this color object</returns>
		public object Clone()
		{
			Color4f c = new Color4f(Red,Green,Blue,Alpha);
			return c;
		}

		#endregion


	}
}
